/* libinklevel.c
 *
 * (c) 2003 Markus Heinz
 *
 * Epson support taken from escputil (gimp-print).
 *
 * This software is licensed under the terms of the GPL.
 * For details see file COPYING.
 */

#include "inklevel.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/poll.h>

#include <ieee1284.h>

#define BUFLEN 1024
#define NR_TAGS 15

#define OLD 1
#define NEW 2

#define IOCNR_GET_DEVICE_ID 1
#define LPIOC_GET_DEVICE_ID _IOC(_IOC_READ, 'P', IOCNR_GET_DEVICE_ID, BUFLEN)

static int get_device_id(int, int, unsigned char *);
static int parse_device_id(int, int, const unsigned char *, 
			   struct ink_level *);
static int parse_device_id_new_hp(unsigned char [NR_TAGS][BUFLEN], int, 
				  struct ink_level *);
static int parse_device_id_old_hp(unsigned char [NR_TAGS][BUFLEN], int, 
				  struct ink_level *);
static int get_ink_level_epson(int, int, int, struct ink_level *);
static int read_from_printer(int, void *, size_t);
static int my_axtoi(unsigned char *);
static int my_atoi(unsigned char *);

int get_ink_level(int port, int portnumber, struct ink_level *level) {
  /*unsigned*/ char device_id[BUFLEN];
  int i;
  int ret;

  for (i = 0; i < MODEL_NAME_LENGTH; i++) {
    level->model[i] = '\0';
  }
  level->type = RESPONSE_INVALID;
  level->black = 0;
  level->color = 0;
  level->cyan = 0;
  level->magenta = 0;
  level->yellow = 0;
  level->photo = 0;
  level->photocyan = 0;
  level->photomagenta = 0;
  level->photoyellow = 0;

  if ((ret = get_device_id(port, portnumber, device_id)) == OK) {
    if ((ret = parse_device_id(port, portnumber, device_id, level)) == OK) {
      return OK;
    }
  }

  return ret;
}

/* This function retrieves the device id of the specified port */

static int get_device_id(int port, int portnumber, unsigned char *device_id) {
  struct parport_list parports;
  unsigned char tmp[BUFLEN];
  unsigned char device_file[256];
  int size;
  int fd;

  if (port == PARPORT ) {
    /* check if we have appropiate permissions */

    sprintf(device_file, "/dev/parport%d", portnumber);
    if ((fd = open(device_file, O_RDWR)) < 0) {
      return DEV_PARPORT_INACCESSIBLE;
    }
    close(fd);

    sprintf(device_file, "/dev/lp%d", portnumber);
    if ((fd = open(device_file, O_RDWR)) < 0) {
      return DEV_LP_INACCESSIBLE;
    }
    close(fd);

    if (ieee1284_find_ports(&parports, 0) == E1284_OK) {
      if (portnumber < parports.portc) {
	size = ieee1284_get_deviceid(parports.portv[portnumber], -1, 
				      F1284_FRESH, tmp, BUFLEN);
	if (size > 0) {
	  strncpy(device_id, tmp + 2, size - 2);
	  return OK;
	}
      }
    }
    return COULD_NOT_GET_DEVICE_ID;
  } else if (port == USB) {
    sprintf(device_file, "/dev/usblp%d", portnumber);
    fd = open(device_file, O_RDONLY);
    if (fd == ERROR) {
      return DEV_USB_LP_INACCESSIBLE;
    }
    if (ioctl(fd, LPIOC_GET_DEVICE_ID, tmp) < 0) {
      close(fd);
      return COULD_NOT_GET_DEVICE_ID;
    }
    close(fd);
    size = (tmp[0] << 8) | (tmp[1]);
    size = (size < BUFLEN - 1) ? size : BUFLEN - 1;
    tmp[size] = '\0';
    strncpy(device_id, tmp + 2, size - 2);
    return OK;
  } else {
    return UNKNOWN_PORT_SPECIFIED;
  }
}

/* This function parses the device id and calls the appropiate function */

static int parse_device_id(int port, int portnumber, 
			   const unsigned char *device_id, 
			   struct ink_level *level) {
  int i = 0;
  int j = 0;
  const unsigned char *c;
  unsigned char tags[NR_TAGS][BUFLEN];  

  /* init tags */

  for (i = 0; i < NR_TAGS; i++) {
    for (j = 0; j < BUFLEN; j++) {
      tags[i][j] = '\0';
    }
  }

  /* Tokenize the device id */

  i = 0;
  j = 0;
  c = device_id;
  
  while ((*c != '\0') && (i < NR_TAGS)) {
    j = 0;
    while ((*c != '\0') && (*c != ';') && (j < BUFLEN)) {
      tags[i][j] = *c;
      c++;
      j++;
    }
    if (*c == ';') { /* Some printers do not terminate the last tag with ';' */
      c++; /* Skip the ';' */
    }
    i++;
  }

#ifdef DEBUG
  /* print all tags */

  for (i = 0; i < NR_TAGS; i++) {
    printf("%d: %s\n", i, tags[i]);
  }
#endif

  /* Check if we deal with a printer */

  /* Find the "CLS:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'C') && (c[1] == 'L') && (c[2] == 'S') && (c[3] == ':')) {
      break;
    }
  }

  if (i < NR_TAGS) {

    /* Check if it is "PRINTER" */

    if ((tags[i][4] != 'P') || (tags[i][5] != 'R') || (tags[i][6] != 'I') || 
	(tags[i][7] != 'N') || (tags[i][8] != 'T') || (tags[i][9] != 'E') || 
	(tags[i][10] != 'R')) {
      
#ifdef DEBUG
      printf("No printer found\n");
#endif
      
      return NO_PRINTER_FOUND;
    }
  } else { 
    /* Find the "CLASS:" tag */
    
    for (i = 0; i < NR_TAGS; i++) {
      c = tags[i];
      if ((c[0] == 'C') && (c[1] == 'L') && (c[2] == 'A') && (c[3] == 'S') && 
	  (c[4] == 'S') && (c[5] == ':')) {
	break;
      }
    }

    if (i < NR_TAGS) {
    
      /* Check if it is "PRINTER" */
      
      if ((tags[i][6] != 'P') || (tags[i][7] != 'R') || (tags[i][8] != 'I') || 
	  (tags[i][9] != 'N') || (tags[i][10] != 'T') || 
	  (tags[i][11] != 'E') || (tags[i][12] != 'R')) {
      
#ifdef DEBUG
	printf("No printer found\n");
#endif
	
	return NO_PRINTER_FOUND;
      }
    } else {

#ifdef DEBUG 
      printf("No device class found\n");
#endif

      return NO_DEVICE_CLASS_FOUND;
    }
  }

  /* Insert the name of the printer */

  /* Find the "MFG:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'M') && (c[1] == 'F') && (c[2] == 'G') && (c[3] == ':')) {
      break;
    }
  }

#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"MFG:\" tag has number %d\n", i);
  } else {
    printf("No \"MFG:\" tag found\n");
  }
#endif

  if (i < NR_TAGS) {
    strcpy(level->model, c + 4);
  }

  j = 0;
  while (level->model[j] != '\0') {
    j++;
  }

  /* Find the "MDL:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'M') && (c[1] == 'D') && (c[2] == 'L') && (c[3] == ':')) {
      break;
    }
  }

#ifdef DEBUG
  if(i < NR_TAGS) {
    printf("The \"MDL:\" tag has number %d\n", i);
  } else {
    printf("No \"MDL:\" tag found\n");
  }
#endif

  if (i < NR_TAGS) {
    level->model[j] = ' ';
    j++;
    strcpy(level->model + j, c + 4);
  }

  /* Check for a new HP printer */

  /* Find the "S:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'S') && (c[1] == ':')) {
      break;
    }
  }

#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"S:\" tag has number %d\n", i);
  } else {
    printf("No \"S:\" tag found\n");
  }
#endif

  if (i<NR_TAGS) {
    return parse_device_id_new_hp(tags, i, level);
  }
  
  /* Check for an old HP printer */

  /* Find the "VSTATUS:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'V') && (c[1] == 'S') && (c[2] == 'T') && (c[3] == 'A') &&
	(c[4] == 'T') && (c[5] == 'U') && (c[6] == 'S') && (c[7] ==':')) {
      break;
    }
  }

#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"VSTATUS:\" tag has number %d\n", i);
  } else {
    printf("No \"VSTATUS:\" tag found\n");
  }
#endif

  if (i<NR_TAGS) {
    return parse_device_id_old_hp(tags, i, level);
  }

  /* Ckeck for a Epson Printer */

  /* Find the "MFG:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'M') && (c[1] == 'F') && (c[2] == 'G') && (c[3] == ':')) {
      break;
    }
  }
    
#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"MFG:\" tag has number %d\n", i);
  } else {
    printf("No \"MFG:\" tag found\n");
  }
#endif

  if (i < NR_TAGS) {
    /* Ckeck if it is "EPSON" */
    c = tags[i];
    if ((c[4] == 'E') && (c[5] == 'P') && (c[6] == 'S') && (c[7] == 'O') &&
	(c[8] == 'N')) {

      /* Check if we deal with an old or a new Epson printer */
      /* A new Epson printer is characterized by "D4" in the "CMD:" tag */
      /* I do not know if this is a viable means of distinction */

      /* Find the "CMD:" tag */

      for (i = 0; i < NR_TAGS; i++) {
	c = tags[i];
	if ((c[0] == 'C') && (c[1] == 'M') && (c[2] == 'D') && (c[3] == ':')) {
	  break;
	}
      }

#ifdef DEBUG
      if (i < NR_TAGS) {
	printf("The \"CMD:\" tag has number %d\n", i);
      } else {
	printf("No \"CMD:\" tag found\n");
      }
#endif

      if (i == NR_TAGS) {
	return NO_CMD_TAG_FOUND;
      }

      while (c[0] != '\0') {
	if ((c[0] == 'D') && (c[1] == '4')) {
	  break;
	}
	c++;
      }

      if (c[0] == '\0') { /* old Epson printer */

#ifdef DEBUG
	printf("Old Epson printer detected\n");
#endif

	return get_ink_level_epson(port, portnumber, OLD, level);
      } else { /* new Epson printer */

#ifdef DEBUG 
	printf("New Epson printer detected\n");
#endif

	return get_ink_level_epson(port, portnumber, NEW, level);
      }	
    }
  }
  
  /* Insert code to check for other printers here */
  
  return PRINTER_NOT_SUPPORTED; /* No matching printer was found */
}

/* This function parses the device id of a new HP printer
 * for example HP Deskjet 5550 
 */

static int parse_device_id_new_hp(unsigned char tags[NR_TAGS][BUFLEN], int n, 
				  struct ink_level *level) {
  unsigned char *s = tags[n];
  int length = 0;
  unsigned char black[3]; /* level of black ink as hexadecimal string */
  unsigned char color[3]; /* level of color ink as hexadecimal string */
  unsigned char photo[3]; /* level of photo ink as hexadecimal string */
  unsigned char cyan[3]; /* level of cyan ink as hexadecimal string */
  unsigned char magenta[3]; /* level of magenta ink as hexadecimal string */
  unsigned char yellow[3]; /* level of yellow ink as hexadecimal string */

  /* Determine the length of the string */

  while (*s != '\0') {
    length++;
    s++;
  }

  s = tags[n];

  /* I do not know for sure if "2c" or "2C" is a viable means of distinction */

  if ((s[length - 17] == '2') && ((s[length - 16] == 'c') || 
                                  (s[length - 16] == 'C'))) {

    /* We deal with a printer with two cartridges */
  
    /* I do not know if "1" always denotes a black cartridge and if "3" always
     * denotes a photo cartridge
     */

    if (s[length - 15] == '1') {

      /* We have black and color cartridge */

      black[0] = s[length - 10];
      black[1] = s[length - 9];
      black[2] = '\0';
      
      color[0] = s[length - 2];
      color[1] = s[length - 1];
      color[2] = '\0';

      level->black = my_axtoi(black);
      level->color = my_axtoi(color);
      level->type = TWO_COLORS_FOUND;

#ifdef DEBUG
      printf("Black: %d%%, Color: %d%%\n", level->black, level->color);
#endif

      return OK;
    } else if (s[length - 15] == '3') {

      /* We have photo and color cartridge */

      photo[0] = s[length - 10];
      photo[1] = s[length - 9];
      photo[2] = '\0';
      
      color[0] = s[length - 2];
      color[1] = s[length - 1];
      color[2] = '\0';

      level->photo = my_axtoi(photo);
      level->color = my_axtoi(color);
      level->type = TWO_PHOTO_COLORS_FOUND;

#ifdef DEBUG
      printf("Color: %d%%, Photo: %d%%\n", level->color, level->photo);
#endif

      return OK;
    }
  } else if ((s[length - 4] == '8') && (s[length - 3] == '0') && 
             (s[length - 8] == '7') && (s[length - 7] == '0') &&
             (s[length - 12] == '6') && (s[length - 11] == '0') &&
             (s[length - 16] == '5') && (s[length - 15] == '0')) {

    /* We deal with a printer with separate colors */
    /* We have black, cyan, magenta and yellow */

    yellow[0] = s[length - 2];
    yellow[1] = s[length - 1];
    yellow[2] = '\0';

    magenta[0] = s[length - 6];
    magenta[1] = s[length - 5];
    magenta[2] = '\0';

    cyan[0] = s[length - 10];
    cyan[1] = s[length - 9];
    cyan[2] = '\0';

    black[0] = s[length - 14];
    black[1] = s[length - 13];
    black[2] = '\0';

    level->yellow = my_axtoi(yellow);
    level->magenta = my_axtoi(magenta);
    level->cyan = my_axtoi(cyan);
    level->black = my_axtoi(black);
    level->type = FOUR_COLORS_FOUND;

#ifdef DEBUG
    printf("Yellow: %d%%, Magenta: %d%%, Cyan: %d%%, Black: %d%%\n",
           level->yellow, level->magenta, level->cyan, level->black);
#endif

    return OK;
  } 

  /* This type of printer is not yet supported */

#ifdef DEBUG
  printf("Printer not supported\n");
#endif

  return PRINTER_NOT_SUPPORTED;
}

/* This function parses the device id of an old HP printer
 * for example HP Photosmart 1000 
 */

static int parse_device_id_old_hp(unsigned char tags[NR_TAGS][BUFLEN], int n, 
				  struct ink_level *level) {
  unsigned char *s = tags[n];
  int length = 0;
  unsigned char b[4]; /* level of black ink as decimal string */
  unsigned char c[4]; /* level of color ink as decimal string */

  /* Determine the length of the string */

  while (*s != '\0') {
    length++;
    s++;
  }

  s = tags[n];
  
  /* Check if "KP" and "CP" tags are there */

  if ((s[length - 11] == 'K') && (s[length - 10] == 'P') &&
      (s[length - 5]  == 'C') && (s[length - 4] == 'P')) {
    b[0] = s[length - 9];
    b[1] = s[length - 8];
    b[2] = s[length - 7];
    b[3] = '\0';
    
    c[0] = s[length - 3];
    c[1] = s[length - 2];
    c[2] = s[length - 1];
    c[3] = '\0';

    level->black = my_atoi(b);
    level->color = my_atoi(c);
    level->type = TWO_COLORS_FOUND;

#ifdef DEBUG
    printf("Black %d%%, Color %d%%\n", level->black, level->color);
#endif

    return OK;
  } else {

#ifdef DEBUG
    printf("No ink level found\n");
#endif
    return NO_INK_LEVEL_FOUND;
  }
}

/* This funtion retrieves the ink level of an Epson printer conncected to
 * the specifies port and portnumber
 */

static int get_ink_level_epson(int port, int portnumber, int type,
			       struct ink_level *level) {
  unsigned char device_file[256];
  int fd;
  unsigned char hdr1[] = "\000\000\000\033\001@EJL 1284.4\n@EJL     \n\033@";
  unsigned char hdr2[] = "\033@\033(R\010\000\000REMOTE1";
  unsigned char cmd[] = "ST\002\000\000\001";
  unsigned char trl[] = "\033\000\000\000\033\000";
  unsigned char rst[] = "\033\000\033\000";
  unsigned char command[256];
  int length;
  int i = 0;
  unsigned char buffer[BUFLEN];
  unsigned char *index = NULL;
  unsigned char number[3];
  int current_pos = 0;

  if (port == USB) {
    sprintf(device_file, "/dev/usb/lp%d", portnumber);
  } else if (port == PARPORT) {
    sprintf(device_file, "/dev/lp%d", portnumber);
  } else {
    return UNKNOWN_PORT_SPECIFIED;
  }

#ifdef DEBUG
  printf("Device file: %s\n", device_file);
#endif

  fd = open(device_file, O_RDWR, 0666);
  if (fd == -1) {

#ifdef DEBUG
    printf("Could not open %s\n", device_file);
#endif

    if (port == USB) {
      return DEV_USB_LP_INACCESSIBLE;
    } else { 
      return DEV_LP_INACCESSIBLE;
    }
  }

  if (type == NEW) { /* we have to exit packet mode first */
    memcpy(command, hdr1, sizeof(hdr1) - 1);
    current_pos += (sizeof(hdr1) - 1);
  }

  memcpy(command + current_pos, hdr2, sizeof(hdr2) - 1);
  current_pos += (sizeof(hdr2) - 1);

  memcpy(command + current_pos, cmd, sizeof(cmd) - 1);
  current_pos += (sizeof(cmd) - 1);

  memcpy(command + current_pos, trl, sizeof(trl) - 1);
  current_pos += (sizeof(trl) - 1);

  memcpy(command + current_pos, rst, sizeof(rst) - 1);
  current_pos += (sizeof(rst) - 1);

  length = current_pos;

#ifdef DEBUG
  printf("Command: \n");
  for(i = 0; i < length; i++) {
    printf("%#4o\n", command[i]);
  }
#endif

  i = write(fd, command, length);
  if (i < length) {

#ifdef DEBUG
    printf("Could not send command to printer\n");
#endif

    close(fd);
    return COULD_NOT_WRITE_TO_PRINTER;
  }

  length = read_from_printer(fd, buffer, BUFLEN);
  if (length <= 0) {

#ifdef DEBUG    
    printf("Could not read from printer\n");
#endif

    close(fd);
    return COULD_NOT_READ_FROM_PRINTER;
  }

#ifdef DEBUG
  printf("Printer returned %s\n", buffer);
#endif

  close(fd);

  index = buffer;

  do {
    index = strchr(index, 'I');
  } while ((index) && (index[1] != 'Q') && (index[1] != '\0') && 
	   (index[2] != ':'));

  if ((!index) || (index[1] != 'Q') || (index[2] != ':')) {

#ifdef DEBUG
    printf("Could not parse output from printer\n");
#endif

    return COULD_NOT_PARSE_RESPONSE_FROM_PRINTER;
  }

  index += 3;

  if ((index[0]) && (index[0] != ';')) {
    level->type = ONE_BLACK_COLOR_FOUND;
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->black = my_axtoi(number);

#ifdef DEBUG
    printf("Black: %d%%\n", level->black);
#endif

    index += 2;
  } else {

#ifdef DEBUG
    printf("No ink level found\n");
#endif

    return NO_INK_LEVEL_FOUND;
  }

  if ((index[0]) && (index[0] != ';')) {
    level->type = FOUR_COLORS_FOUND;
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->cyan = my_axtoi(number);

#ifdef DEBUG
    printf("Cyan: %d%%\n", level->cyan);
#endif

    index += 2;
  }

  if ((index[0]) && (index[0] != ';')) {
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->magenta = my_axtoi(number);

#ifdef DEBUG
    printf("Magenta: %d%%\n", level->magenta);
#endif

    index += 2;
  }

  if ((index[0]) && (index[0] != ';')) {
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->yellow = my_axtoi(number);

#ifdef DEBUG
    printf("Yellow: %d%%\n", level->yellow);
#endif

    index += 2;
  }

  if ((index[0]) && (index[0] != ';')) {
    level->type = SIX_COLORS_FOUND;
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->photocyan = my_axtoi(number);

#ifdef DEBUG
    printf("Photocyan: %d%%\n", level->photocyan);
#endif

    index += 2;
  }
  if ((index[0]) && (index[0] != ';')) {
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->photomagenta = my_axtoi(number);

#ifdef DEBUG
    printf("Photomagenta: %d%%\n", level->photomagenta);
#endif

    index += 2;
  }
  if ((index[0]) && (index[0] != ';')) {
    level->type = SEVEN_COLORS_FOUND;
    number[0] = index[0];
    number[1] = index[1];
    number[2] = '\0';
    level->photoyellow = my_axtoi(number);

#ifdef DEBUG
    printf("Photoyellow: %d%%\n", level->photoyellow);
#endif

  }

  return OK;
}

/* This function reads from the printer nonblockingly */

static int read_from_printer(int fd, void *buf, size_t bufsize) {
  int status;
  int retry = 5;
  struct pollfd ufds;

  fcntl(fd, F_SETFL, O_NONBLOCK | fcntl(fd, F_GETFL));
  memset(buf, 0, bufsize);

  do {
    ufds.fd = fd;
    ufds.events = POLLIN;
    ufds.revents = 0;
    if ((status = poll(&ufds, 1, 1000)) < 0) {
      break;
    }
    status = read(fd, buf, bufsize);
    if ((status == 0) || ((status < 0) && (errno == EAGAIN))) {
      sleep(1);
      status = 0;
    }
  } while ((status == 0) && (--retry != 0));

#ifdef DEBUG
  if ((status == 0) && (retry == 0)) {
    printf("Read from printer timed out\n");
  } else if (status < 0) {
    printf("Could not read from printer\n");
  }
#endif

  return status;
}


/* This function converts a string containing a two digit hexadecimal number
 * to an int 
 */

static int my_axtoi(unsigned char* t) {
  int r = 0;

  switch (t[0]) {
  case '0':
    r = 0;
    break;
  case '1':
    r = 16;
    break;
  case '2':
    r = 32;
    break;
  case '3':
    r = 48;
    break;
  case '4':
    r = 64;
    break;
  case '5':
    r = 80;
    break;
  case '6':
    r = 96;
    break;
  case '7':
    r = 112;
    break;
  case '8':
    r = 128;
    break;
  case '9':
    r = 144;
    break;
  case 'a':
  case 'A':
    r = 160;
    break;
  case 'b':
  case 'B':
    r = 176;
    break;
  case 'c':
  case 'C':
    r = 192;
    break;
  case 'd':
  case 'D':
    r = 208;
    break;
  case 'e':
  case 'E':
    r = 224;
    break;
  case 'f':
  case 'F':
    r = 240;
    break;
  }

  switch (t[1]) {
  case '0':
    r += 0;
    break;
  case '1':
    r += 1;
    break;
  case '2':
    r += 2;
    break;
  case '3':
    r += 3;
    break;
  case '4':
    r += 4;
    break;
  case '5':
    r += 5;
    break;
  case '6':
    r += 6;
    break;
  case '7':
    r += 7;
    break;
  case '8':
    r += 8;
    break;
  case '9':
    r += 9;
    break;
  case 'a':
  case 'A':
    r += 10;
    break;
  case 'b':
  case 'B':
    r += 11;
    break;
  case 'c':
  case 'C':
    r += 12;
    break;
  case 'd':
  case 'D':
    r += 13;
    break;
  case 'e':
  case 'E':
    r += 14;
    break;
  case 'f':
  case 'F':
    r += 15;
    break;
  }

  return r;
}

/* This function converts a string containig a three digit decimal number
 * to an int 
 */

static int my_atoi(unsigned char* t) {
  int r = 0;

  switch (t[0]) {
  case '0':
    r = 0;
    break;
  case '1':
    r = 100;
    break;
  case '2':
    r = 200;
    break;
  case '3':
    r = 300;
    break;
  case '4':
    r = 400;
    break;
  case '5':
    r = 500;
    break;
  case '6':
    r = 600;
    break;
  case '7':
    r = 700;
    break;
  case '8':
    r = 800;
    break;
  case '9':
    r = 900;
    break;
  }
  
  switch (t[1]) {
  case '0':
    r += 0;
    break;
  case '1':
    r += 10;
    break;
  case '2':
    r += 20;
    break;
  case '3':
    r += 30;
    break;
  case '4':
    r += 40;
    break;
  case '5':
    r += 50;
    break;
  case '6':
    r += 60;
    break;
  case '7':
    r += 70;
    break;
  case '8':
    r += 80;
    break;
  case '9':
    r += 90;
    break;
  }

  switch (t[2]) {
  case '0':
    r += 0;
    break;
  case '1':
    r += 1;
    break;
  case '2':
    r += 2;
    break;
  case '3':
    r += 3;
    break;
  case '4':
    r += 4;
    break;
  case '5':
    r += 5;
    break;
  case '6':
    r += 6;
    break;
  case '7':
    r += 7;
    break;
  case '8':
    r += 8;
    break;
  case '9':
    r += 9;
    break;
  }
  
  return r;
}
