/* ink.c
 *
 * (c) 2003 Markus Heinz
 *
 * This software is licensed under the terms of the GPL.
 * For details see file COPYING.
 */

#include <inklevel.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

void usage(void) {
  printf("ink -p \"usb\"|\"parport\" [-n <portnumber>]\n");
}

int main(int argc, char *argv[]) {
  struct ink_level *level = NULL;
  int result = 0;
  int port = 0;
  int portnumber = 0;
  char c;

  printf("ink v0.3 (c) 2003 Markus Heinz\n\n");

  if (argc == 1) {
    usage();
    return 1;
  }

  while ((c = getopt(argc, argv, "p:n:")) != -1) {
    switch (c) {
    case 'p':
      if (strcmp(optarg, "parport") == 0) {
	port = PARPORT;
      } else if (strcmp(optarg, "usb") == 0) {
	port = USB;
      } else {
	usage();
	return 1;
      }
      break;
    case 'n':
      if (optarg) {
	portnumber = atoi(optarg);
      } else {
	usage();
	return 1;
      }
      break;
    default:
      usage();
      return 1;
    }
  }

  level = (struct ink_level *) malloc(sizeof(struct ink_level));

  if (level == NULL) {
    printf("Not enough memory available.\n");
    return 1;
  }
  
  result = get_ink_level(port, portnumber, level);

  printf("%s\n\n", level->model);

  if (result != OK) {
    switch (result) {
    case DEV_PARPORT_INACCESSIBLE:
      printf("Could not access '/dev/parport%d'.\n", portnumber);
      break;
    case DEV_LP_INACCESSIBLE:
      printf("Could not access '/dev/lp%d'.\n", portnumber);
      break;
    case COULD_NOT_GET_DEVICE_ID:
      printf("Could not get device id.\n");
      break;
    case DEV_USB_LP_INACCESSIBLE:
      printf("Could not access '/dev/usb/lp%d'.\n", portnumber);
      break;
    case UNKNOWN_PORT_SPECIFIED:
      printf("Unknown port specified.\n");
      break;
    case NO_PRINTER_FOUND:
      printf("No printer found.\n");
      break;
    case NO_DEVICE_CLASS_FOUND:
      printf("No device class found.\n");
      break;
    case NO_CMD_TAG_FOUND:
      printf("No cmd tag found.\n");
      break;
    case PRINTER_NOT_SUPPORTED:
      printf("Printer not supported.\n");
      break;
    case NO_INK_LEVEL_FOUND:
      printf("No ink level found.\n");
      break;
    case COULD_NOT_WRITE_TO_PRINTER:
      printf("Could not write to printer.\n");
      break;
    case COULD_NOT_READ_FROM_PRINTER:
      printf("Could not read from printer.\n");
      break;
    case COULD_NOT_PARSE_RESPONSE_FROM_PRINTER:
      printf("Could not parse response from printer.\n");
      break;
    }
    printf("Could not get ink level.\n");
    free(level);
    return 1;
  }

  switch (level->type) {
  case RESPONSE_INVALID:
    printf("No ink level found\n");
    break;
  case ONE_COLOR_FOUND:
    printf("%-13s %3d%%\n", "Color:", level->color);
    break;
  case ONE_BLACK_COLOR_FOUND:
    printf("%-13s %3d%%\n", "Black:", level->black);
    break;
  case TWO_COLORS_FOUND:
    printf("%-13s %3d%%\n", "Black:", level->black);
    printf("%-13s %3d%%\n", "Color:", level->color);
    break;
  case TWO_PHOTO_COLORS_FOUND:
    printf("%-13s %3d%%\n", "Photo:", level->photo);
    printf("%-13s %3d%%\n", "Color:", level->color);
    break;
  case THREE_COLORS_FOUND:
    printf("%-13s %3d%%\n", "Cyan:", level->cyan);
    printf("%-13s %3d%%\n", "Magenta:", level->magenta);
    printf("%-13s %3d%%\n", "Yellow:", level->yellow);
    break;
  case FOUR_COLORS_FOUND:
    printf("%-13s %3d%%\n", "Black:", level->black);
    printf("%-13s %3d%%\n", "Cyan:", level->cyan);
    printf("%-13s %3d%%\n", "Magenta:", level->magenta);
    printf("%-13s %3d%%\n", "Yellow:", level->yellow);
    break;
  case SIX_COLORS_FOUND:
    printf("%-13s %3d%%\n", "Black:", level->black);
    printf("%-13s %3d%%\n", "Cyan:", level->cyan);
    printf("%-13s %3d%%\n", "Magenta:", level->magenta);
    printf("%-13s %3d%%\n", "Yellow:", level->yellow);
    printf("%-13s %3d%%\n", "Photocyan:", level->photocyan);
    printf("%-13s %3d%%\n", "Photomagenta:", level->photomagenta);
    break;
  case SEVEN_COLORS_FOUND:
    printf("%-13s %3d%%\n", "Black:", level->black);
    printf("%-13s %3d%%\n", "Cyan:", level->cyan);
    printf("%-13s %3d%%\n", "Magenta:", level->magenta);
    printf("%-13s %3d%%\n", "Yellow:", level->yellow);
    printf("%-13s %3d%%\n", "Photocyan:", level->photocyan);
    printf("%-13s %3d%%\n", "Photomagenta:", level->photomagenta);
    printf("%-13s %3d%%\n", "Photoyellow:", level->photoyellow);
    break;
  }

  free(level);

  return 0;
}
