
/**
 *
 * Tennix! SDL Port
 * Copyright (C) 2003, 2007 Thomas Perl <thp@perli.net>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 *
 **/

#include <stdio.h>
#include <math.h>

#include "tennix.h"
#include "game.h"
#include "graphics.h"
#include "input.h"
#include "sound.h"

void game( bool singleplayer) {
    GameState s = {
        { 0, 0, 0.0, 0.0, 0.0 },
        { 0, 0, 0, 0, 0 },
        { GAME_X_MIN-RACKET_X_MID*2, GAME_Y_MID, 0, 0, 1, DESIRE_NORMAL, PLAYER_TYPE_HUMAN, false },
        { GAME_X_MAX+RACKET_X_MID*2, GAME_Y_MID, 0, 0, 0, DESIRE_NORMAL, PLAYER_TYPE_HUMAN, false },
        0,
        0,
        0,
        true,
        "welcome to tennix " VERSION,
        { 0 },
        { 0 },
        REFEREE_NORMAL,
        0,
        WINNER_NONE,
        false,
        GR_CTT_GRASS
    };

    strcpy( s.game_score_str, format_game( &s));
    strcpy( s.sets_score_str, format_sets( &s));

    Uint32 ot = SDL_GetTicks();
    Uint32 nt;
    Uint32 dt = GAME_TICKS;
    Uint32 diff;
    Uint32 accumulator = 0;
    bool quit = false;

    if( singleplayer) {
#ifdef DEBUG
        s.player1.type = PLAYER_TYPE_AI;
#endif
        s.player2.type = PLAYER_TYPE_AI;
    }

    game_setup_serve( &s);
    sound_audience();
   
    while( !quit) {
        nt = SDL_GetTicks();
        diff = nt-ot;
        if( diff > 2000) {
            diff = 0;
        }

        accumulator += diff;
        ot = nt;

        while( accumulator >= dt) {
            quit = step( &s);
            s.time += dt;
            accumulator -= dt;

            if( s.was_stopped) {
                ot = SDL_GetTicks();
                s.was_stopped = false;
            }
        }

        render( &s);
    }
   
}

bool step( GameState* s) {
    Uint8 *keys;
    SDL_Event e;
   
    if( get_phase( s) < 1.0) {
        if( !s->ground.jump) {
            sound_ground();

            if( IS_OUT_Y( s->ball.y)) {
                /* out - responsibilities stay the same */
                s->status = "out!";
                sound_out();
                s->referee = REFEREE_OUT;
            } else {
                /* not out - responsibilities change */
                s->player1.responsible = !(s->player2.responsible = !s->player2.responsible);
                s->status = format_status( s);
                s->referee = REFEREE_NORMAL;
            }
        }
        s->ground.jump = 3;
        s->ground.x = s->ball.x;
        s->ground.y = s->ball.y;
    } else {
        if( s->ground.jump && !(s->time%5)) s->ground.jump--;
    }

    if( IS_OUT_X(s->ball.x) || IS_OFFSCREEN_Y(s->ball.y)) {
        if( IS_OFFSCREEN( s->ball.x, s->ball.y)) {
            s->player1_serves = s->player1.responsible;

            score_game( s, s->player2.responsible);
            strcpy( s->game_score_str, format_game( s));
            strcpy( s->sets_score_str, format_sets( s));

            if( s->player1.responsible) {
                if( s->player1.type == PLAYER_TYPE_HUMAN && s->player2.type == PLAYER_TYPE_AI) {
                    s->status = "computer scores";
                } else {
                    s->status = "player 2 scores";
                }
                s->referee = REFEREE_PLAYER2;
            } else {
                if( s->player1.type == PLAYER_TYPE_HUMAN && s->player2.type == PLAYER_TYPE_AI) {
                    s->status = "player scores";
                } else {
                    s->status = "player 1 scores";
                }
                s->referee = REFEREE_PLAYER1;
            }

            game_setup_serve( s);
            sound_applause();
            SDL_Delay( 500);
            start_fade();
            s->was_stopped = true;
        }
    
        if( IS_OUT_X(s->ball.x)) {
            if( s->ball.move_x <= 0 && IS_NEAR_X( s->player1.x, s->ball.x) && IS_NEAR_Y( s->player1.y, s->ball.y) && s->player1.state && s->referee != REFEREE_OUT) {
                s->ball.x = GAME_X_MIN;
                if( s->player1.state == PLAYER_STATE_MAX) {
                    s->ball.move_x = PLAYER_POWERSHOT;
                } else {
                    s->ball.move_x = 2.5 + 2.0*s->player1.state/PLAYER_STATE_MAX;
                }
                s->ball.move_y = get_move_y( s, 1);
                s->player2.responsible = !(s->player1.responsible = 1);
                s->ball.jump += 1.0-2.0*(s->player1.state<5);
                sound_applause_stop();
                sound_racket();
            } else if( s->ball.move_x >= 0 && IS_NEAR_X( s->player2.x, s->ball.x) && IS_NEAR_Y( s->player2.y, s->ball.y) && s->player2.state && s->referee != REFEREE_OUT) {
                s->ball.x = GAME_X_MAX;
                if( s->player2.state == PLAYER_STATE_MAX) {
                    s->ball.move_x = -PLAYER_POWERSHOT;
                } else {
                    s->ball.move_x = -(2.5 + 2.0*s->player2.state/PLAYER_STATE_MAX);
                }
                s->ball.move_y = get_move_y( s, 2);
                s->player1.responsible = !(s->player2.responsible = 1);
                s->ball.jump += 1.0-2.0*(s->player2.state<5);
                sound_applause_stop();
                sound_racket();
            }
        }
    }

    /* Update ball_dest for debugging purposes */
    get_move_y( s, 1);
    get_move_y( s, 2);

    s->ball.x += s->ball.move_x;
    s->ball.y += s->ball.move_y;
    
    if( s->player1.state) s->player1.state--;
    if( s->player2.state) s->player2.state--;

    SDL_PollEvent( &e);
    keys = SDL_GetKeyState( NULL);

    if( keys['c'] && s->time%50==0) {
        s->court_type++;
        if( s->court_type > GR_CTT_LAST) {
            s->court_type = GR_CTT_FIRST;
        }
    }

    if( !is_fading() && !s->is_over) {
        if( s->player1.type == PLAYER_TYPE_HUMAN) {
           input_human( &s->player1, keys['w'], keys['s'], keys['d']);
        } else {
            input_ai( &s->player1, &s->ball, &s->player2);
        }
 
        if( s->player2.type == PLAYER_TYPE_HUMAN) {
            input_human( &s->player2, keys['o'], keys['l'], keys['k']);
        } else {
            input_ai( &s->player2, &s->ball, &s->player1);
        }
    }
    
    if( keys['f']) SDL_WM_ToggleFullScreen( screen);
    if( keys['y']) SDL_SaveBMP( screen, "screenshot.bmp");
     
    if( keys[SDLK_ESCAPE] || keys['q']) return true;
    
    limit_value( &s->player1.y, PLAYER_Y_MIN, PLAYER_Y_MAX);
    limit_value( &s->player2.y, PLAYER_Y_MIN, PLAYER_Y_MAX);
    limit_value( &s->ball.jump, BALL_JUMP_MIN, BALL_JUMP_MAX);
    
    return false;
}

void render( GameState* s) {
    if( s->winner != WINNER_NONE) {
        if( !s->is_over) {
            start_fade();
            s->is_over = true;
        }
        clearscr();
        show_sprite( GR_RACKET, 2*(s->winner-1), 4, WIDTH/2 - get_image_width( GR_RACKET)/8, HEIGHT/2 - get_image_height( GR_RACKET), 255);
        sprintf( s->game_score_str, "player %d wins the match with %s", s->winner, format_sets( s));
        font_draw_string( GR_DKC2_FONT, s->game_score_str, (WIDTH-font_get_string_width( GR_DKC2_FONT, s->game_score_str))/2, HEIGHT/2 + 30, s->time/20, ANIMATION_WAVE | ANIMATION_BUNGEE);
        updatescr();
        return;
    }
    clearscr();
    fill_image( s->court_type, 120, 120, 400, 250);
    show_image( GR_COURT, 0, 0, 255);
    show_sprite( GR_REFEREE, s->referee, 4, 250, 10, 255);
    show_image( GR_SHADOW, s->ball.x-BALL_X_MID, s->ball.y + get_phase( s) - BALL_Y_MID, 255);
    
    show_sprite( GR_RACKET, (!s->player1.state), 4, s->player1.x-RACKET_X_MID, s->player1.y-RACKET_Y_MID, 255);
    show_sprite( GR_RACKET, (!s->player2.state)+2, 4, s->player2.x-RACKET_X_MID, s->player2.y-RACKET_Y_MID, 255);
    
    if( s->ground.jump) {
        show_sprite( GR_GROUND, s->ground.jump-1, 3, s->ground.x - BALL_X_MID, s->ground.y - BALL_Y_MID, 128);
    }

    if( s->ball.move_x > 0) {
        show_sprite( GR_BALL, (s->time/500)%4, 4, s->ball.x-BALL_X_MID, s->ball.y-BALL_Y_MID, 255);
    } else if( s->ball.move_x < 0) {
        show_sprite( GR_BALL, 3-(s->time/500)%4, 4, s->ball.x-BALL_X_MID, s->ball.y-BALL_Y_MID, 255);
    } else {
        show_sprite( GR_BALL, 0, 4, s->ball.x-BALL_X_MID, s->ball.y-BALL_Y_MID, 255);
    }

    
    font_draw_string( GR_DKC2_FONT, s->game_score_str, 14, 14, 0, ANIMATION_NONE);
    font_draw_string( GR_DKC2_FONT, s->sets_score_str, (WIDTH-font_get_string_width( GR_DKC2_FONT, s->sets_score_str))-14, 14, 0, ANIMATION_NONE);

    font_draw_string( GR_DKC2_FONT, s->status, (WIDTH-font_get_string_width( GR_DKC2_FONT, s->status))/2, HEIGHT-50, s->time/30, ANIMATION_WAVE);

#ifdef DEBUG
    line_horiz( s->player1.y, 255, 0, 0);
    line_horiz( s->player2.y, 0, 0, 255);
    line_horiz( s->ball.y, 0, 255, 0);

    line_vert( s->player1.x, 255, 0, 0);
    line_vert( s->player2.x, 0, 0, 255);
    line_vert( s->ball.x, 0, 255, 0);

    line_horiz( s->player1.ball_dest, 255, 0, 255);
    line_horiz( s->player2.ball_dest, 0, 255, 255);

    line_horiz( GAME_Y_MIN, 100, 100, 100);
    line_horiz( GAME_Y_MAX, 100, 100, 100);
#endif

    updatescr();
}

void limit_value( float* value, float min, float max) {
    if( *value < min) {
        *value = min;
    } else if( *value > max) {
        *value = max;
    }
}

float get_phase( GameState* s) {
    float pos, fract;
    float x, min, max, direction;

    x = s->ball.x;
    min = GAME_X_MIN;
    max = GAME_X_MAX;
    direction = s->ball.move_x;

    pos = (direction>0)?(1-GROUND_PHASE):(GROUND_PHASE);

    fract = (x-min)/(max-min);

    if( fract < pos) {
        fract = fract/pos;
        return fabsf( cosf(PI*fract/2))*PHASE_AMP*s->ball.jump;
    } else {
        fract = (pos-fract)/(1-pos);
        return fabsf( sinf(PI*fract/2))*PHASE_AMP*s->ball.jump;
    }
}

float get_move_y( GameState* s, unsigned char player) {
    float pct, dest, x_len, y_len;
    float py, by, pa, move_x;

    py = (player==1)?(s->player1.y):(s->player2.y);
    by = s->ball.y;
    pa = RACKET_Y_MID*2;
    move_x = s->ball.move_x;

    /* -1.0 .. 1.0 for racket hit position */
    pct = (by-py)/(pa/2);
    limit_value( &pct, -1.0, 1.0);

    /* Y destination for ball */
    dest = GAME_Y_MID + pct*(GAME_Y_MAX-GAME_Y_MIN);
    if( player == 1) {
        s->player1.ball_dest = dest;
    } else {
        s->player2.ball_dest = dest;
    }

    /* lengths for the ball's journey */
    if( player == 1) {
        x_len = fabsf(GAME_X_MAX - s->ball.x);
        y_len = dest - by + MOVE_Y_SEED-rand()%MOVE_Y_SEED*2;
    } else {
        x_len = s->ball.x - GAME_X_MIN;
        y_len = by - dest + MOVE_Y_SEED-rand()%MOVE_Y_SEED*2;
    }

    /* return the should-be value for move_y */
    return (y_len*move_x)/(x_len);
}

void input_human( Player* player, bool up, bool down, bool hit) {
    if( up) {
        player->y -= 6;
    }

    if( down) {
        player->y += 6;
    }

    if( hit) {
        if( !player->state && !player->state_locked) {
            player->state = PLAYER_STATE_MAX;
            player->state_locked = true;
        }
    } else {
        player->state_locked = false;
    }
}

void input_ai( Player* player, Ball* ball, Player* opponent) {
    float fact = 1.5;
    float target;

    if( fabsf( player->y - ball->y) > RACKET_Y_MID*5) {
        fact = 4;
    }

    target = GAME_Y_MID + (opponent->ball_dest - GAME_Y_MID)/10;

    if( player->responsible) {
        if( player->desire == DESIRE_NORMAL && !IS_NEAR_Y_AI( player->y, ball->y)) {
            if( player->y < ball->y) {
                player->y += fmin( 2*fact, ball->y - player->y);
            } else if( player->y > ball->y) {
                player->y -= fmin( 2*fact, player->y - ball->y);
            }
        }

        if( (ball->move_x != 0 || IS_NEAR_Y_AI( player->y, ball->y)) && IS_NEAR_X_AI( player->x, ball->x) && !player->state && rand()%4) {
            player->state = PLAYER_STATE_MAX;
        }
    } else if( ball->move_x == 0) {
        if( player->desire == DESIRE_NORMAL && !IS_NEAR_Y_AI( player->y, target)) {
            if( player->y < target) {
                player->y += fmin( fact, (target-player->y)/40.0);
            } else if( player->y > target) {
                player->y -= fmin( fact, (player->y-target)/40.0);
            }
        }
    } else {/*
        if( player->desire == DESIRE_NORMAL) {
            if( !IS_NEAR_Y_AI( player->y, target)) {
                player->y += (target - player->y)/40.0;
            }
        }*/
    }
}

void game_setup_serve( GameState* s) {
    s->ball.jump = 7.5;
    s->ball.y = GAME_Y_MID;
    s->ball.move_x = 0.0;
    s->ball.move_y = 0.0;

    if( s->player1_serves) {
        s->player1.responsible = true;
        s->player1.ball_dest = 0.0;
        s->ball.x = GAME_X_MIN-RACKET_X_MID*1.5;
    } else {
        s->player1.responsible = false;
        s->player2.ball_dest = 0.0;
        s->ball.x = GAME_X_MAX+RACKET_X_MID*1.5;
    }

    s->player2.responsible = !(s->player1.responsible);
}

void score_game( GameState* s, bool player1_scored) {
    Player* winner = (player1_scored)?(&(s->player1)):(&(s->player2));
    Player* loser = (player1_scored)?(&(s->player2)):(&(s->player1));

    if( s->current_set >= SETS_TO_WIN*2-1) {
        return;
    }

    winner->game++;
    if( loser->game < winner->game-1) {
        if( winner->game >= 4) {
            winner->game = loser->game = 0;
            winner->sets[s->current_set]++;
            /* scoring the set.. */
            if( (winner->sets[s->current_set] == 6 && loser->sets[s->current_set] < 5) ||
                winner->sets[s->current_set] == 7) {
                s->current_set++;
                s->winner = game_get_winner( s);
            }
        }
    }
}

char* format_sets( GameState* s) {
    static char sets[100];
    static char tmp[100];
    int i, max = s->current_set;

    sets[0] = '\0';

    if( s->winner != WINNER_NONE) {
        max--;
    }
    for( i=0; i<=max; i++) {
        sprintf( tmp, "%d:%d, ", s->player1.sets[i], s->player2.sets[i]);
        strcat( sets, tmp);
    }

    sets[strlen(sets)-2] = '\0';

    return sets;
}

char* format_game( GameState* s) {
    static char game[100];
    static const int game_scoring[] = { 0, 15, 30, 40 };

    if( s->player1.game < 4 && s->player2.game < 4) {
        sprintf( game, "%d - %d", game_scoring[s->player1.game], game_scoring[s->player2.game]);
    } else if( s->player1.game > s->player2.game) {
        strcpy( game, "advantage player 1");
    } else if( s->player1.game < s->player2.game) {
        strcpy( game, "advantage player 2");
    } else {
        strcpy( game, "deuce");
    }

    return game;
}

char* format_status( GameState* s) {
    static char status[100];
    static const char* set_names[] = { "first", "second", "third", "fourth", "fifth" };

    sprintf( status, "%d:%d in %s set", s->player1.sets[s->current_set], s->player2.sets[s->current_set], set_names[s->current_set]);

    return status;
}

int game_get_winner( GameState* s) {
    int i;
    int sets[2] = {0};

    for( i=0; i<s->current_set; i++) {
        if( s->player1.sets[i] > s->player2.sets[i]) {
            sets[0]++;
        } else {
            sets[1]++;
        }
    }

    if( sets[0] == SETS_TO_WIN) return WINNER_PLAYER1;
    if( sets[1] == SETS_TO_WIN) return WINNER_PLAYER2;

    return WINNER_NONE;
}

