#ifndef __SCENEGRAPH_H__
#define __SCENEGRAPH_H__

#include <osg/MatrixTransform>
#include <osg/PolygonMode>
#include <osg/ShadeModel>
#include <osg/LineStipple>
#include "osgtreemodel.h"
#include "orthomanipulator.h"

/** @class SceneGraph
 * @brief OSGEdit Scenegraph.
 *
 * It encapsulates actually two OSG scenegraphs. The one loaded by the user
 * and the one that OSGEdit uses to show things in the screen. As you may
 * notice, osgedit shows the selected node with duplicated geometry in
 * wireframe, and has a grid. These are osg nodes in a higher level scenegraph
 * that contains osgedit-specific nodes and the scenegraph of the user.
 */
class SceneGraph: public osg::Referenced {
    osg::ref_ptr<osg::Group> root; ///< A root node with childs like the file, the selection, etc.
    osg::ref_ptr<osg::Node> file; ///< The file really opened by the user
    osg::ref_ptr<osg::MatrixTransform> selection; ///< A node displaying the current selection
    osg::ref_ptr<osg::Group> selection_decorator; // Decorator for the selected node showing it in wireframe
    osg::ref_ptr<osg::MatrixTransform> axis; ///< XYZ axis
    osg::ref_ptr<osg::LineStipple> selection_stipple; /// REMOVE! The stipple, to make an effect ;)
    osg::ref_ptr<OsgTreeModel> treemodel; ///< The tree model to notify the user interface of changes in the scenegraph
    osg::ref_ptr<osg::StateSet> global_state_set; ///< A global state set to change things like solid/wireframe, smooth/flat...
    osg::ref_ptr<osg::MatrixTransform> gridNode; ///< the grid node!
    osg::NodePath selected; ///< Path to the selected node
    osg::Matrix selected_parents_transform; ///< Transform until the parent of the selected node
public:
    SceneGraph();
    osg::Node *loadFile(const char *file); ///< Loads the root node from the file and returns it
    void saveFile(const char *file);
    void replaceScenegraph(osg::Node *newSG); ///< Replaces the current scenegraph with another one

    void selectNodePath(osg::NodePath &path, bool notifyUI=true); ///< This path is selected!
    void recalculateSelection(); ///< Recalculates the selection boundings
    osg::Matrix getParentTransform() { return selected_parents_transform; }

    // Configuring the model
    void setPolygonMode(osg::PolygonMode::Mode m);
    void setShadeModel(osg::ShadeModel::Mode m);

    // Tree manipulation API
    void addChild(osg::Group *parent, osg::Node *child);
    void removeChild(osg::Group *parent, osg::Node *child);
    void replaceChild(osg::Group *parent, osg::Node *oldNode, osg::Node *newNode);
    void renameNode(osg::Node *node, const char *name);

    // Some usefull getters
    osg::MatrixTransform* getGridNode() { return gridNode.get(); }

    // Must not be used, I will try to delete them
    osg::Node* getRealRoot() { return root.get(); }
    OsgTreeModel *getTreeModel() { return treemodel.get(); }
    void refresh();
    osg::Group *makeValidParent(osg::Group *g) {
	return g!=0?g:root.get();
    }
    bool isRoot(osg::Node *node) { return node == file.get(); }
    void setRoot(osg::Node *root);
    void setTreeModel(OsgTreeModel *t) { treemodel=t; }
    osg::Node* getRoot() { return file.get(); }
private:
};

#endif
