#include <config.h>
#include <osg/Geometry>
#include <osg/Geode>
#include <osg/LineWidth>
#include <osg/Material>
#include <osg/PolygonOffset>
#include <osgDB/ReadFile>
#include <osgDB/WriteFile>
#include "scenegraph.h"
#include "clipboard.h"
#include "local.h"
#include <locale.h>
#include "appconfig.h"

#include <gui/warning.h>
#include <command/comdelete.h>
#include <command/comaddchild.h>
#include <command/comaddtransform.h>
#include "editor.h"

osg::MatrixTransform *makeGrid() {
    osg::MatrixTransform *transform=new osg::MatrixTransform();
    osg::Geode *geode=new osg::Geode();
    osg::Geometry *geom=new osg::Geometry();
    osg::Vec3Array *vertices=new osg::Vec3Array();
    int i;
    CfgGrid *grid=AppConfig::instance()->getCfgGrid();
    int numLines=(int)(grid->getEdgeSize()/grid->getUnitSize())/2;
    float radius=grid->getEdgeSize()/2;
    vertices->push_back(osg::Vec3(0.0f,-radius,0.0f));
    vertices->push_back(osg::Vec3(0.0f, radius,0.0f));
    vertices->push_back(osg::Vec3(-radius,0.0f,0.0f));
    vertices->push_back(osg::Vec3( radius,0.0f,0.0f));
    for (i=1;i<numLines;i++) {
	vertices->push_back(osg::Vec3(i*grid->getUnitSize(),-radius,0.0f));
	vertices->push_back(osg::Vec3(i*grid->getUnitSize(),radius,0.0f));
	vertices->push_back(osg::Vec3(-i*grid->getUnitSize(),-radius,0.0f));
	vertices->push_back(osg::Vec3(-i*grid->getUnitSize(),radius,0.0f));
	vertices->push_back(osg::Vec3(-radius,i*grid->getUnitSize(),0.0f));
	vertices->push_back(osg::Vec3(radius,i*grid->getUnitSize(),0.0f));
	vertices->push_back(osg::Vec3(-radius,-i*grid->getUnitSize(),0.0f));
	vertices->push_back(osg::Vec3(radius,-i*grid->getUnitSize(),0.0f));
    }
    geom->setVertexArray(vertices);
    osg::Vec4Array* colors = new osg::Vec4Array;
    colors->push_back(grid->getAxisColor());
    colors->push_back(grid->getGridColor());
    geom->setColorArray(colors);
    geom->setColorBinding(osg::Geometry::BIND_PER_PRIMITIVE_SET);

    osg::Vec3Array* normals = new osg::Vec3Array;
    normals->push_back(osg::Vec3(0.0f,-1.0f,0.0f));
    geom->setNormalArray(normals);
    geom->setNormalBinding(osg::Geometry::BIND_OVERALL);

    geom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::LINES,0,4));
    geom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::LINES,4,vertices->size()-4));
    geode->addDrawable(geom);
    osg::StateSet *set=new osg::StateSet();
    set->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
    geode->setStateSet(set);
    transform->addChild(geode);
    return transform;
}

osg::Group *makeSelection(osg::LineStipple *stipple) {
    osg::Group* decorator = new osg::Group;
    // set up the state so that the underlying color is not seen through
    // and that the drawing mode is changed to wireframe, and a polygon offset
    // is added to ensure that we see the wireframe itself, and turn off 
    // so texturing too.
    osg::StateSet* stateset = new osg::StateSet;
    osg::Material* material = new osg::Material;
    osg::PolygonOffset* polyoffset = new osg::PolygonOffset;
    polyoffset->setFactor(-1.0f);
    polyoffset->setUnits(-1.0f);
    osg::PolygonMode* polymode = new osg::PolygonMode;
    polymode->setMode(osg::PolygonMode::FRONT_AND_BACK,osg::PolygonMode::LINE);
    stateset->setAttributeAndModes(material,osg::StateAttribute::OVERRIDE|osg::StateAttribute::ON);
    stateset->setAttributeAndModes(polyoffset,osg::StateAttribute::OVERRIDE|osg::StateAttribute::ON);
    stateset->setAttributeAndModes(polymode,osg::StateAttribute::OVERRIDE|osg::StateAttribute::ON);
    stateset->setMode(GL_LIGHTING,osg::StateAttribute::OVERRIDE|osg::StateAttribute::OFF);
    stateset->setTextureMode(0,GL_TEXTURE_2D,osg::StateAttribute::OVERRIDE|osg::StateAttribute::OFF);
    
    decorator->setStateSet(stateset);

    return decorator;
}

osg::MatrixTransform *makeAxis() {
    osg::MatrixTransform *transform=new osg::MatrixTransform();
    osg::Geode *geode=new osg::Geode();
    osg::Geometry *geom=new osg::Geometry();
    osg::Vec3Array *vertices=new osg::Vec3Array();

    vertices->push_back(osg::Vec3(0,0,0));
    vertices->push_back(osg::Vec3(1,0,0));

    vertices->push_back(osg::Vec3(0,0,0));
    vertices->push_back(osg::Vec3(0,1,0));

    vertices->push_back(osg::Vec3(0,0,0));
    vertices->push_back(osg::Vec3(0,0,1));

    geom->setVertexArray(vertices);
    osg::Vec4Array* colors = new osg::Vec4Array;
    colors->push_back(osg::Vec4(1.0f,0.0f,0.0f,1.0f));
    colors->push_back(osg::Vec4(0.0f,1.0f,0.0f,1.0f));
    colors->push_back(osg::Vec4(0.0f,0.0f,1.0f,1.0f));
    geom->setColorArray(colors);
    geom->setColorBinding(osg::Geometry::BIND_PER_PRIMITIVE);

    geom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::LINES,0,2));
    geom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::LINES,2,2));
    geom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::LINES,4,2));
    geode->addDrawable(geom);
    osg::StateSet *set=new osg::StateSet();
    set->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
    osg::LineWidth *width=new osg::LineWidth();
    width->setWidth(2.0f);
    set->setAttributeAndModes(width, osg::StateAttribute::ON);
    geode->setStateSet(set);
    transform->addChild(geode);
    return transform;
}

SceneGraph::SceneGraph() {
    root= new osg::Group();
    file=0;
    osg::LineStipple *stipple=new osg::LineStipple();
    selection=new osg::MatrixTransform();//makeSelection(&stipple);
    selection_decorator=makeSelection(stipple);
    selection->addChild(selection_decorator.get());
    root->addChild(selection.get());
    axis=makeAxis();
    root->addChild(axis.get());
    gridNode=makeGrid();
    root->addChild(gridNode.get());
    global_state_set=new osg::StateSet();
    root->setStateSet(global_state_set.get());
    osg::Matrix m;
    m.makeScale(0,0,0);
    selection->setMatrix(m);
}

osg::Node* SceneGraph::loadFile(const char *filename) {
    char *previous=setlocale(LC_ALL,NULL);
    setlocale(LC_ALL,"C");
    osg::Node *n=osgDB::readNodeFile(filename);
    setlocale(LC_ALL,previous);
    return n;
}

void SceneGraph::replaceScenegraph(osg::Node *newSG) {
    if (file.valid()) {
        root->removeChild(file.get());
    }
    file=newSG;
    root->addChild(file.get());
    treemodel->setRootNode(file.get());
    History::clear();
}

void SceneGraph::setRoot(osg::Node *root) {
    file=root;
}

void SceneGraph::refresh() {
    treemodel->setRootNode(file.get());
}

void SceneGraph::saveFile(const char *filen) {
    osg::Node *n=file.get();
    if (n==0) {
        my_warning(_("There is nothing to save"));
    } else {
	char *previous=setlocale(LC_ALL,NULL);
	setlocale(LC_ALL,"C");
	osgDB::writeNodeFile(*n,filen);
	setlocale(LC_ALL,previous);
    }
}

void SceneGraph::setPolygonMode(osg::PolygonMode::Mode m) {
    osg::PolygonMode* polyModeObj = new osg::PolygonMode;
    polyModeObj->setMode(osg::PolygonMode::FRONT_AND_BACK,m);
    global_state_set->setAttribute(polyModeObj);
}

void SceneGraph::setShadeModel(osg::ShadeModel::Mode m) {
    osg::StateSet* stateset = global_state_set.get();
    osg::ShadeModel* shademodel = dynamic_cast<osg::ShadeModel*>(stateset->getAttribute(osg::StateAttribute::SHADEMODEL));
    if (!shademodel)
    {
	shademodel = new osg::ShadeModel;
	stateset->setAttribute(shademodel,osg::StateAttribute::OVERRIDE);
    }
    shademodel->setMode(m);
}

void SceneGraph::addChild(osg::Group *parent, osg::Node *child) {
    if (parent) {
	parent->addChild(child);
        treemodel->addChild(parent,child);
    } else {
        throw -1;
    }
}

void SceneGraph::removeChild(osg::Group *parent, osg::Node *child) {
    if (parent) {
	parent->removeChild(child);
        treemodel->removeChild(parent,child);
    } else {
        throw -1;
    }
}

void SceneGraph::replaceChild(osg::Group *parent, osg::Node *oldNode, osg::Node *newNode) {
    if (parent) {
        parent->replaceChild(oldNode,newNode);
        treemodel->removeChild(parent,oldNode);
        treemodel->addChild(parent,newNode);
    } else {
        throw -1;
    }
}

void SceneGraph::renameNode(osg::Node *node, const char *name) {
    if (node) {
	node->setName(name);
        treemodel->renameNode(node,name);
    } else {
        throw -1;
    }
}

class MyVisitor: public osg::NodeVisitor {
    osg::Matrix accumulated;
    osg::MatrixTransform *selection;
    osg::MatrixTransform *axis;
    osg::NodePath toTest;
public:
    MyVisitor(osg::MatrixTransform *sel, osg::MatrixTransform *axis, osg::NodePath &path) {
	this->selection=sel;
        this->axis=axis;
        this->toTest=path;
    }
    bool test(osg::Node &node) {
	osg::NodePath other = getNodePath();
	if (toTest == other) {
	    osg::Matrix trans;
	    osg::Matrix scale;
	    trans.makeTranslate(node.getBound().center());
            //float r=node.getBound().radius();
	    //scale.makeScale(r,r,r);
            getLocalToWorldMatrix(accumulated, &node);
	    osg::Matrix result=scale*trans*accumulated;
	    selection->setMatrix(accumulated);
	    osg::Matrix onlyTranslate;
            onlyTranslate.makeTranslate(result(3,0), result(3,1), result(3,2));
            axis->setMatrix(onlyTranslate);
	    return true;
	} else {
            return false;
	}
    }
    osg::Matrix getAccumulated() { return accumulated; }
    virtual void apply(osg::Node& node) {
	test(node);
    }

    virtual void apply(osg::Group& node) {
	if (!test(node)) {
	    for (unsigned i=0;i<node.getNumChildren();i++) {
                node.getChild(i)->accept(*this);
	    }
	}
    }
};

void SceneGraph::selectNodePath(osg::NodePath &path, bool notifyUI) {
    selected=path;
    if (path.size() > 0) {
	osg::NodePath::iterator iter = path.end() -1;
	osg::Node *node = *iter;
	if (selection_decorator->getNumChildren() > 0) {
	    selection_decorator->removeChild(selection_decorator->getChild(0));
	}
	selection_decorator->addChild(node);
    } else {
	if (selection_decorator->getNumChildren() > 0) {
	    selection_decorator->removeChild(selection_decorator->getChild(0));
	}
    }
    if (notifyUI) {
	treemodel->select(path);
    }
}

void SceneGraph::recalculateSelection() {
    MyVisitor v(selection.get(), axis.get(), selected);
    if (file.valid()) {
	file->accept(v);
        selected_parents_transform=v.getAccumulated();
    }
}
