#include "gtkviewer.h"

#include <osgGA/TrackballManipulator>
#include <osgGA/FlightManipulator>
#include <osgGA/DriveManipulator>
#include <osgUtil/DisplayRequirementsVisitor>
#include "gtkeventadapter.h"

GtkViewer::GtkViewer() {
    _displaySettings=new osg::DisplaySettings();
    _frameStamp=new osg::FrameStamp();
    _focusedViewport=0;
    _initialTime=_timer.tick();
}

GtkViewer::~GtkViewer() {

}

void GtkViewer::addViewport(osgUtil::SceneView* sv,
			    float x     , float y      ,
			    float width , float height ) {
    ViewportDef def;
    def.sceneView   = sv;
    def.viewport[0] = x;
    def.viewport[1] = y;
    def.viewport[2] = width;
    def.viewport[3] = height;
    _viewportList.push_back(def);
}

void GtkViewer::addViewport(osg::Node* rootnode,
			    float x     , float y      ,
			    float width , float height ) {
    osgUtil::SceneView *sceneView = new osgUtil::SceneView(_displaySettings.get());
    sceneView->setDefaults();
    sceneView->setSceneData(rootnode);

    addViewport( sceneView, x, y, width, height );
}


bool GtkViewer::open() {

    if ( getNumViewports() <= 0 ) {
	osg::notify(osg::FATAL)<<"osgGLUT::Viewer::open() called with no Viewports registered."<< std::endl;
	return false;
    }
    // Verify all viewports have an active camera manipulator
    unsigned int index = 0;
    ViewportList::iterator itr;
    for(itr=_viewportList.begin();
	itr!=_viewportList.end();
	++itr, ++index)
    {
	if (itr->_cameraManipList.empty())
	{
	    osg::notify(osg::INFO)<<"osgGLUT::Viewer::open() called without any camara manipulators registered for a viewport,"<< std::endl;
	    osg::notify(osg::INFO)<<"automatically registering trackball,flight and drive manipulators."<< std::endl;
	    registerCameraManipulator(new osgGA::TrackballManipulator, index);
	    registerCameraManipulator(new osgGA::FlightManipulator, index);
	    registerCameraManipulator(new osgGA::DriveManipulator, index);
	}

	if (!itr->_cameraManipulator.valid())
	    selectCameraManipulator(0,index);
    }

    bool needQuadBufferStereo = false;

    // do we need quad buffer stereo?

    // Set the absolute viewport for each SceneView based on the
    //   relative viewport coordinates given to us
    for(itr=_viewportList.begin();
	itr!=_viewportList.end();
	++itr)
    {
	osgUtil::SceneView* sceneView = itr->sceneView.get();
	int view[4] = { int(itr->viewport[0]*_width), int(itr->viewport[1]*_height),
	int(itr->viewport[2]*_width), int(itr->viewport[3]*_height) };

	sceneView->setViewport(view[0], view[1], view[2], view[3]);

/*	osg::ref_ptr<GLUTEventAdapter> ea = new GLUTEventAdapter;
	ea->adaptResize(clockSeconds(),
			view[0], view[1],
			view[0]+view[2], view[1]+view[3]);

	if (itr->_cameraManipulator->handle(*ea,*this))
	{
	    //        osg::notify(osg::INFO) << "Handled reshape "<< std::endl;
	}
*/
	const osg::DisplaySettings* vs = sceneView->getDisplaySettings();
	if (vs)
	{
	    _displaySettings->merge(*vs);
	}
	else
	{
	    // one does not already exist so attach the viewers visualsSettins to the SceneView
	    sceneView->setDisplaySettings(_displaySettings.get());
	}

    }

    if (_displaySettings->getStereo() &&
	_displaySettings->getStereoMode()==osg::DisplaySettings::QUAD_BUFFER) needQuadBufferStereo = true;

    // traverse the scene graphs gathering the requirements of the OpenGL buffers.
    osgUtil::DisplayRequirementsVisitor drv;
    drv.setDisplaySettings(_displaySettings.get());
    for(itr=_viewportList.begin();
	itr!=_viewportList.end();
	++itr)
    {
	osg::Node* node = itr->sceneView->getSceneData();
	if (node) node->accept(drv);
    }

    // set up each render stage to clear the appropriate buffers.
    GLbitfield clear_mask=0;
    if (_displaySettings->getRGB())              clear_mask |= GL_COLOR_BUFFER_BIT;
    if (_displaySettings->getDepthBuffer())      clear_mask |= GL_DEPTH_BUFFER_BIT;
    if (_displaySettings->getStencilBuffer())    clear_mask |= GL_STENCIL_BUFFER_BIT;

    for(itr=_viewportList.begin();
	itr!=_viewportList.end();
	++itr)
    {
	osgUtil::RenderStage *stage = itr->sceneView->getRenderStage();
	stage->setClearMask(clear_mask);
    }

        // Reset the views of all of SceneViews
    osg::ref_ptr<GtkEventAdapter> ea = new GtkEventAdapter;

    for(itr=_viewportList.begin();
        itr!=_viewportList.end();
        ++itr)
    {
        itr->_cameraManipulator->home(*ea,*this);
    }

    return GtkGLWidget::open();
}

// called on each frame redraw..return the time in ms for each operation.
float GtkViewer::app(unsigned int viewport) {
    osg::ref_ptr<GtkEventAdapter> ea = new GtkEventAdapter();
    ea->adaptFrame(_frameStamp->getReferenceTime());

    _viewportList[viewport]._cameraManipulator->handle(*ea,*this);

    getViewportSceneView(viewport)->setFrameStamp(_frameStamp.get());
    getViewportSceneView(viewport)->update();
    return 0.0f;
}
float GtkViewer::cull(unsigned int viewport) {
    getViewportSceneView(viewport)->cull();
    return 0.0f;
}

float GtkViewer::draw(unsigned int viewport) {
    //glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,_two_sided_lighting);
    getViewportSceneView(viewport)->draw();
    return 0.0f;
}

// handle multiple camera.
unsigned int GtkViewer::registerCameraManipulator(osgGA::CameraManipulator* cm,
						  unsigned int viewport) {
    ViewportDef &viewp = _viewportList[viewport];
    unsigned int pos   = viewp._cameraManipList.size();
    viewp._cameraManipList.push_back(cm);
    return pos;
}
void GtkViewer::selectCameraManipulator(unsigned int pos,
					unsigned int viewport) {
    if (viewport>=_viewportList.size()) return;

    ViewportDef &viewp = _viewportList[viewport];
    if (pos>=viewp._cameraManipList.size()) return;

    viewp._cameraManipulator = viewp._cameraManipList[pos];

    osgUtil::SceneView *sceneView = viewp.sceneView.get();
    viewp._cameraManipulator->setCamera(sceneView->getCamera());
    viewp._cameraManipulator->setNode(sceneView->getSceneData());

    osg::ref_ptr<GtkEventAdapter> ea = new GtkEventAdapter;
    viewp._cameraManipulator->init(*ea,*this);

}

GtkEventAdapter *GtkViewer::makeEvent() {
    GtkEventAdapter *ea = new GtkEventAdapter();
    ViewportDef &vp=_viewportList[_focusedViewport];
    int view[4] = {
	int(vp.viewport[0]*_width), int(vp.viewport[1]*_height),
	int(vp.viewport[2]*_width), int(vp.viewport[3]*_height) };
    ea->adaptResize(0,
		    view[0], view[1],
		    view[0]+view[2], view[1]+view[3]);
    return ea;
}

void GtkViewer::mouseRelease(int button) {
    GtkGLWidget::mouseRelease(button);
    osg::ref_ptr<GtkEventAdapter> ea = makeEvent();
    ea->adaptMouseRelease(0,_buttonMask,_mx,_my);
    _viewportList[_focusedViewport]._cameraManipulator->handle(*ea,*this);
}
void GtkViewer::mousePress(int button) {
    GtkGLWidget::mousePress(button);
    osg::ref_ptr<GtkEventAdapter> ea = makeEvent();
    ea->adaptMousePress(0,_buttonMask,_mx,_my);
    _viewportList[_focusedViewport]._cameraManipulator->handle(*ea,*this);
}

void GtkViewer::mouseMotion(double x, double y) {
    GtkGLWidget::mouseMotion(x,y);
    osg::ref_ptr<GtkEventAdapter> ea = makeEvent();
    ea->adaptMouseMove(0,_buttonMask,x,y);
    _viewportList[_focusedViewport]._cameraManipulator->handle(*ea,*this);
    int focus = mapWindowXYToSceneView((int)x,(int)y);
    if (focus >= 0 && focus != int(_focusedViewport))
	setFocusedViewport(focus);
}

void GtkViewer::reshape(double width, double height) {
    GtkGLWidget::reshape(width,height);

    // Propagate new window size to viewports
    for(ViewportList::iterator itr=_viewportList.begin();
	itr!=_viewportList.end();
        ++itr)
    {
        osgUtil::SceneView* sceneView = itr->sceneView.get();
        int view[4] = { int(itr->viewport[0]*_width), int(itr->viewport[1]*_height),
                        int(itr->viewport[2]*_width), int(itr->viewport[3]*_height) };

        sceneView->setViewport(view[0], view[1], view[2], view[3]);

	osg::ref_ptr<GtkEventAdapter> ea = new GtkEventAdapter;
        ea->adaptResize(0,
                        view[0], view[1],
			view[0]+view[2], view[1]+view[3]);
        itr->_cameraManipulator->handle(*ea,*this);

/*        osg::ref_ptr<GLUTEventAdapter> ea = new GLUTEventAdapter;
        ea->adaptResize(clockSeconds(),
                        view[0], view[1],
                        view[0]+view[2], view[1]+view[3]);

        if (itr->_cameraManipulator->handle(*ea,*this))
        {
            //        osg::notify(osg::INFO) << "Handled reshape "<< std::endl;
        }*/
    }

}

void GtkViewer::display() {
    _frameStamp->setFrameNumber(_frameStamp->getFrameNumber()+1);
    _frameStamp->setReferenceTime(clockSeconds());
    for(int i = 0; i < getNumViewports(); i++ )
    {
	// application traverasal.
	app(i);
	// cull traverasal.
	cull(i);
	// draw traverasal.
	draw(i);
    }
    GtkGLWidget::swapBuffers();
}

void GtkViewer::requestWarpPointer(float x,float y) {}

void GtkViewer::setFocusedViewport(unsigned int pos) {
    if (pos>=_viewportList.size()) return;

    _focusedViewport = pos;
}

int GtkViewer::mapWindowXYToSceneView(int x, int y) {
    int ogl_y = _height-y;

    int index = 0;
    for(ViewportList::iterator itr=_viewportList.begin();
        itr!=_viewportList.end();
        ++itr, ++index)
    {
      if ( x >= int( itr->viewport[0]*_width ) &&
           ogl_y  >= int( itr->viewport[1]*_height ) &&
           x <= int( (itr->viewport[0]+itr->viewport[2])*_width ) &&
           ogl_y  <= int( (itr->viewport[1]+itr->viewport[3])*_height ) )
        return index;
    }
    return -1;

}
