#ifndef __EDITOR_H__
#define __EDITOR_H__

class Tool;
#include <string>
#include <map>
#include "cameraview.h"
#include "scenegraph.h"
#include <command/history.h>

/** @class Editor
 * @brief Representation of the generic editor controller
 *
 * This class must be subclassed on each concrete editor (for example, in
 * the default GNOME one). This is part of the strategy followed in OSGEdit
 * to allow for multiple views (Model-View-Controller pattern). If you want
 * to "port" OSGEdit to another platform where there is no GNOME, the first
 * thing is to subclass Editor.
 */
class Editor: public osg::Referenced, public HistoryObserver {
    /* Private data, don't worry about this */
    osg::ref_ptr<Tool> activeTool; ///< The currently active tool
    osg::ref_ptr<CameraView> view; ///< The view of the editor, for now only one is supported
    osg::ref_ptr<osg::Node> selectedNode; ///< The currently selected node
    osg::ref_ptr<osg::Group> selectedParent; ///< The parent of the currently selected node
    std::string loaded_file; ///< The name of the loaded file, to know where to save
protected:
    osg::ref_ptr<SceneGraph> sceneGraph; ///< The OSGEdit scenegraph

    /// Here you will receive events when the user selects one node to highlight its name or something
    virtual void selected(osg::Node *node, osg::NodePath &path)=0;
public:
    /// Your interface must call this method in order to activate a tool
    void useTool(char *toolname);

    /// Call here if you want to select a node (for tools that operate on nodes)
    void selectNode(osg::Node *node, osg::NodePath &path, bool notifyUI=true);

    /// Returns the currently selected node
    osg::Node *getSelected() { return selectedNode.get(); }

    /// Returns the camera view representing the point of view in osgedit
    CameraView *getCameraView() { return view.get(); }

    /// Returns the OSGEdit Scenegraph
    SceneGraph *getSceneGraph() { return sceneGraph.get(); }

    /**
     * Here you will receive events when some command is executed. It is
     * not needed to implement this method, but is usefull to update the
     * interface as this command could have changed the scenegraph or
     * the attributes of some node */
    virtual void executed(Command *c) {}

    /// Called when some command is un-executed, see executed for details
    virtual void unexecuted(Command *c) {}

    /**
     * As the Editor is osg::Referenced, you can have problems when
     * terminating the application. Use this method to solve this problems
     * for example freeing timers (is the use in GNOME interface) */
    virtual void finalize() {}

    /// Changes the name of the loaded file
    void setLoadedFile(const char *filename) { loaded_file=filename; }

    /// Gets the name of the loaded file
    const char *getLoadedFile() { return loaded_file.c_str(); }

    /**
     * Gets the path to the loaded file. Needed for example, to figure out
     * where the textures are placed in the filesystem and show them in
     * your toolkit's image widget */
    std::string getLoadedFilePath();

    /** Allow any tool to ask the editing of some object */
    virtual void editObject(osg::Object *)=0;
protected: // Ensure nobody instantiates this class
    Editor();
};

#endif
