#include <osg/Texture2D>
#include <osg/TextureCubeMap>

#include "edstateset.h"
#include <command/comstateset.h>

static std::string getGLName(osg::StateAttribute::GLMode mode) {
    switch (mode) {
    case GL_ALPHA_TEST: return "GL_ALPHA_TEST";
    case GL_BLEND: return "GL_BLEND";
    case GL_COLOR_MATERIAL: return "GL_COLOR_MATERIAL";
    case GL_CULL_FACE: return "GL_CULL_FACE";
    case GL_DEPTH_TEST: return "GL_DEPTH_TEST";
    case GL_FOG: return "GL_FOG";
    case GL_LIGHTING: return "GL_LIGHTING";
    case GL_POINT_SMOOTH: return "GL_POINT_SMOOTH";
    case GL_LINE_STIPPLE: return "GL_LINE_STIPPLE";
    case GL_POLYGON_OFFSET_FILL: return "GL_POLYGON_OFFSET_FILL";
    case GL_POLYGON_OFFSET_LINE: return "GL_POLYGON_OFFSET_LINE";
    case GL_POLYGON_OFFSET_POINT: return "GL_POLYGON_OFFSET_POINT";
    case GL_COLOR_SUM: return "GL_COLOR_SUM";
    case GL_TEXTURE_1D: return "GL_TEXTURE_1D";
    case GL_TEXTURE_2D: return "GL_TEXTURE_2D";
    case GL_TEXTURE_3D: return "GL_TEXTURE_3D";

    case GL_TEXTURE_CUBE_MAP: return "GL_TEXTURE_CUBE_MAP";

    case GL_TEXTURE_GEN_Q: return "GL_TEXTURE_GEN_Q";
    case GL_TEXTURE_GEN_R: return "GL_TEXTURE_GEN_R";
    case GL_TEXTURE_GEN_S: return "GL_TEXTURE_GEN_S";
    case GL_TEXTURE_GEN_T: return "GL_TEXTURE_GEN_T";

    case GL_STENCIL_TEST: return "GL_STENCIL_TEST";

    case GL_CLIP_PLANE0: return "GL_CLIP_PLANE0";
    case GL_CLIP_PLANE1: return "GL_CLIP_PLANE1";
    case GL_CLIP_PLANE2: return "GL_CLIP_PLANE2";
    case GL_CLIP_PLANE3: return "GL_CLIP_PLANE3";
    case GL_CLIP_PLANE4: return "GL_CLIP_PLANE4";
    case GL_CLIP_PLANE5: return "GL_CLIP_PLANE5";

    case GL_LIGHT0: return "GL_LIGHT0";
    case GL_LIGHT1: return "GL_LIGHT1";
    case GL_LIGHT2: return "GL_LIGHT2";
    case GL_LIGHT3: return "GL_LIGHT3";
    case GL_LIGHT4: return "GL_LIGHT4";
    case GL_LIGHT5: return "GL_LIGHT5";
    case GL_LIGHT6: return "GL_LIGHT6";
    case GL_LIGHT7: return "GL_LIGHT7";
    default: {
	char s[1024];
	sprintf(s, "0x%x", mode);
	return std::string(s);
    }
    }
}



void EdStateSet::setObject(osg::Object *object) {
    model=dynamic_cast<osg::StateSet *> (object);
    if (model == 0) {
	std::cerr << "EdStateSet::setObject(): FATAL ERROR: This is not an 'osg::StateSet'" << std::endl;
	throw -1;
    }
    modes = new EdList( _("Modes"), _("Mode of each state attribute of the set") );

    osg::StateSet::ModeList mlist = model->getModeList();
    osg::StateSet::ModeList::iterator miter;
    for (miter=mlist.begin();miter != mlist.end();miter++) {
	EditableAttributeList list;
	osg::ref_ptr<EdBool> on;
	osg::ref_ptr<EdBool> override;
	osg::ref_ptr<EdBool> _protected;
	on = new EdBool( _("On"), _("Enabled/Disabled"), miter->second & osg::StateAttribute::ON );

	list.push_back(on.get());
	override = new EdBool( _("Override"), _("This attribute will try to override attributes on child nodes"), miter->second & osg::StateAttribute::OVERRIDE );

	list.push_back(override.get());
	_protected = new EdBool( _("Protected"), _("This attribute can't be overriden by any parent's attribute"), miter->second & osg::StateAttribute::PROTECTED );

	list.push_back(_protected.get());
	modes->add(getGLName(miter->first),list);
    }

    list.push_back(modes.get());
    texture_modes = new EdList( _("TextureModes"), _("Mode of each texture state attribute of the set") );

    osg::StateSet::TextureModeList tmlist = model->getTextureModeList();
    osg::StateSet::ModeList::iterator tmiter;
    for (unsigned unit = 0; unit < tmlist.size(); unit++) {
	for (tmiter=tmlist[unit].begin();tmiter != tmlist[unit].end();tmiter++) {
	    EditableAttributeList list;
	    osg::ref_ptr<EdBool> on;
	    osg::ref_ptr<EdBool> override;
	    osg::ref_ptr<EdBool> _protected;
	    on = new EdBool( _("On"), _("Enabled/Disabled"), tmiter->second & osg::StateAttribute::ON );

	    list.push_back(on.get());
	    override = new EdBool( _("Override"), _("This texture attribute will try to override attributes on child nodes"), tmiter->second & osg::StateAttribute::OVERRIDE );

	    list.push_back(override.get());
	    _protected = new EdBool( _("Protected"), _("This texture attribute can't be overriden by any parent's attribute"), tmiter->second & osg::StateAttribute::PROTECTED );

	    list.push_back(_protected.get());
	    char s[1024];
            sprintf(s,"%s<%d>",getGLName(tmiter->first).c_str(),unit);
	    texture_modes->add(s,list);
	}
    }

    list.push_back(texture_modes.get());
    rendering_hint = new EdEnum( _("Rendering hint"), _("Specify here in which renderbin will this stateset fall") );
    rendering_hint->addOption(_("Default"),osg::StateSet::DEFAULT_BIN);
    rendering_hint->addOption(_("Opaque"),osg::StateSet::OPAQUE_BIN);
    rendering_hint->addOption(_("Transparent"),osg::StateSet::TRANSPARENT_BIN);
    rendering_hint->setInitial(model->getRenderingHint());

    list.push_back(rendering_hint.get());

}

EdStateSet::EdStateSet() {}

EdStateSet::EdStateSet(osg::StateSet *_model) {
    setObject(_model);
}

void EdStateSet::commit() {
    osg::StateSet::ModeList _modes;
    unsigned i;
    osg::StateSet::ModeList mlist = model->getModeList();
    osg::StateSet::ModeList::iterator miter;
    for (miter=mlist.begin(),i=0;miter != mlist.end();miter++,i++) {
	const EditableAttributeList list =modes->get(i);
        const EdBool *ed_on = dynamic_cast<const EdBool*>(list[0].get());
	bool is_on = ed_on->get();
        const EdBool *ed_override = dynamic_cast<const EdBool*>(list[1].get());
	bool is_override = ed_override->get();
        const EdBool *ed_protected = dynamic_cast<const EdBool*>(list[2].get());
	bool is_protected = ed_protected->get();
	osg::StateAttribute::GLModeValue value=is_on*osg::StateAttribute::ON
	    | is_override*osg::StateAttribute::OVERRIDE
	    | is_protected*osg::StateAttribute::PROTECTED;

	_modes[miter->first]=value;
    }

    osg::StateSet::TextureModeList tmlist = model->getTextureModeList();
    osg::StateSet::ModeList::iterator tmiter;

    osg::StateSet::TextureModeList _texture_modes(tmlist.size());

    for (unsigned unit = 0,i=0; unit < tmlist.size(); unit++) {
	for (tmiter=tmlist[unit].begin();tmiter != tmlist[unit].end();tmiter++,i++) {
	    const EditableAttributeList list =texture_modes->get(i);
	    const EdBool *ed_on = dynamic_cast<const EdBool*>(list[0].get());
	    bool is_on = ed_on->get();
	    const EdBool *ed_override = dynamic_cast<const EdBool*>(list[1].get());
	    bool is_override = ed_override->get();
	    const EdBool *ed_protected = dynamic_cast<const EdBool*>(list[2].get());
	    bool is_protected = ed_protected->get();
	    osg::StateAttribute::GLModeValue value=is_on*osg::StateAttribute::ON
		| is_override*osg::StateAttribute::OVERRIDE
		| is_protected*osg::StateAttribute::PROTECTED;

	    _texture_modes[unit][tmiter->first]=value;
	}
    }
    osg::ref_ptr<Command> com =
	new ComStateSet(model.get(),
			_modes,
			_texture_modes,
			(int)rendering_hint->get());
    com->execute();
}
