#ifndef __ED_REFERENCE_H__
#define __ED_REFERENCE_H__

#include <editable/editableattribute.h>
#include <map>

class EditableEntity;

/** @class EdReference
 * @brief Editable attribute for references to entities
 *
 * It has a binded list of classes, to understand the meaning of this,
 * see an example:
 *  Imagine that you have a class A with a reference to a class B. But
 *  this class B is abstract, with two concrete subclasses C, D. Now
 *  you want to give the user the posibility of change either the attributes
 *  of the actual referenced object, or even change the referenced object
 *  itself, to be of another class. That is, imagine, that your instance of
 *  A (a), has a reference to an instance of C (c), but user wants to change
 *  a to use an instance of D instead. This is the reason why EdReference has
 *  a list of classes, they are in fact prototypes of all the possible
 *  classes allowed. This way, you can change this reference to use another
 *  referenced class. Polymorphism is good ;)
 *
 *  A more concrete example: You have class Employ, which has a reference
 *  to an Enterprise. But Enterprise is abstract, with subclasses Provider
 *  and Client. Now you open your application and you find that some Employ
 *  is binded to a Provider, when in fact should be binded with a Client.
 *  You can ask EdReference to point to an instance of Client.
 *
 * Anyway, it's possible to have a reference to some instance and don't allow
 * changing it. Simply don't register classes. User wan't be able to change
 * the class of the referenced object. The interface should reflect this by
 * disabling the buttons used for this task.
 */
class EdReference: public EditableAttribute {
public:
    EdReference(std::string name, std::string description);

    /** Returns the type of this attribute */
    virtual std::string getType() { return "reference"; }

    inline EditableEntity* get() { return value.get(); }
    inline EditableEntity* getOriginal() { return original.get(); }

    /// Adds a new class supported by this reference
    void addClass(std::string name, EditableEntity *prototype);

    /// Instantiates one of the added prototypes as the actual reference
    void set(std::string the_class);

    /// Sets initial pointer
    void setInitial(EditableEntity *entity);

    /// Sets the reference to null
    void setNull();

    /** Checks if the attribute has changed */
    inline virtual bool hasChanged() const { return original != value; }

    virtual std::string toString() const;

    std::vector<std::string> getClasses();
private:
    osg::ref_ptr<EditableEntity> value;
    osg::ref_ptr<EditableEntity> original;
    std::map<std::string, osg::ref_ptr<EditableEntity> > map;
};

#endif
