#ifndef __ED_ENUM_H__
#define __ED_ENUM_H__

#include <string>
#include <editable/editableattribute.h>

/** @class EdEnum
 * @brief Editable attribute for enumerations
 */
class EdEnum: public EditableAttribute {
public:
    EdEnum(std::string name, std::string description);

    /** Returns the type of this attribute */
    virtual std::string getType() { return "enum"; }

    /** Returns the real value of the selected option */
    int getValue() const;

    /** Returns the name of the selected option */
    std::string getValueName() const;

    /** Returns the index in the option list of the selected option */
    inline int getValueIndex() const { return value_index; }

    /** Returns the name of the selected option */
    std::string getOriginalValueName() const;

    /** Returns the index in the option list of the selected option */
    inline int getOriginalValueIndex() const { return original_index; }

    /** Returns the *real* value */
    inline int get() const { return options[value_index].second; }

    /** Changes the selection. Sets 'changed' flag.
     * @param _value_index The index of the selected option, NOT the real value
     */
    void set(int _value_index);

    /** Establishes the *real* selected value for the first time. Doesn't set
     * the 'changed' flag.
     * @param _value The real value. Must have some option for it or this
     * method will fail!
     */
    void setInitial(int _value);

    /** Adds a new option with it's binded value. Call this before setInitial
     * for each option, to ensure that the class can find the option for
     * the values...
     */
    void addOption(std::string name, int value);

    /** Checks if the attribute has changed */
    virtual bool hasChanged() const { return original_index != value_index; }

    /** Returns current number of options */
    inline unsigned getNumOptions() const { return options.size(); }

    /** Return one of the option strings */
    std::string getOption(unsigned i) const;

    inline virtual std::string toString() const { return getValueName(); }
private:
    int value_index;
    int original_index;
    typedef std::pair<std::string, int> EnumOption;
    typedef std::vector<EnumOption> OptionList;
    OptionList options;
};

#endif
