#include "cameraview.h"
#include <osg/Matrix>
#define DEFAULT_BACKGROUND_COLOR 0.5,0.5,0.5,1.0

#define CAMERA_DISTANCE 20

#ifndef PI
#  define PI 3.14159265359
#endif

CameraView::CameraView() {
    this->zoom=0.3f;
    this->sceneView=new osgUtil::SceneView();
    sceneView->setDefaults();
    sceneView->setBackgroundColor(osg::Vec4(DEFAULT_BACKGROUND_COLOR));
    applyViewType();

    center_perspective=osg::Vec3(0,0,0);
    orbit_angle_perspective = osg::Vec2(0,0);
    projection=ORTHOGONAL;
    axis_system=Y_UP;
}

void CameraView::setData(osg::Node *node) {
    sceneView->setSceneData(node);
}

void CameraView::setBackgroundColor(osg::Vec4 v) {
    sceneView->setBackgroundColor(v);
}

void CameraView::applyTrackball() {
    osg::Matrix mH=osg::Matrix::rotate(orbit_angle_perspective[0],0,1,0);
    osg::Matrix mP=osg::Matrix::rotate(-orbit_angle_perspective[1],1,0,0);
    osg::Matrix transf=mP*mH;
    osg::Matrix translate=osg::Matrix::translate(center_perspective);
    osg::Camera *cam=sceneView->getCamera();

    osg::Vec3 eye(0,0,CAMERA_DISTANCE);
    eye=eye*transf*translate;

    osg::Vec3 center(0,0,-1);
    center=center*transf+eye;

    osg::Vec3 up(0,1,0);
    up=up*transf;
    cam->setLookAt(eye,center,up);
}

#define ROT_90_RAD 1.570796327
#define SQR(x) ((x)*(x))
#define RAD2DEG(x) ( (x)*180/3.14159265356 )

void CameraView::applyViewType() {
    osg::Camera *cam=sceneView->getCamera();
    applyTrackball();
    switch (projection) {
    case ORTHOGONAL:
	cam->setOrtho2D(-1.0f/zoom,1.0f/zoom,-1.0f/zoom,1.0f/zoom);
	break;
    case PERSPECTIVE:
	cam->setPerspective(2*RAD2DEG(atanf(1.5f/zoom/CAMERA_DISTANCE)),1.0f, 1.0f,200.0f);
	break;
    }
}

void CameraView::setGridNode(osg::MatrixTransform *gridnode) {
    this->gridNode=gridnode;
    applyViewType();
}

void CameraView::rotateCamera(float h_angle, float p_angle) {
    orbit_angle_perspective+=osg::Vec2(h_angle,p_angle);
    if (gridNode.valid()) {
	osg::Matrix matrix;
	matrix.makeRotate(ROT_90_RAD,1,0,0);
	gridNode->setMatrix(matrix);
    }
    applyViewType();
}

void CameraView::moveCamera(float x_offset, float y_offset, float z_offset) {
    center_perspective+=osg::Vec3(x_offset, y_offset, z_offset);
    applyViewType();
}

void CameraView::setCenter(float x_center, float y_center, float z_center) {
    center_perspective=osg::Vec3(x_center, y_center, z_center);
    applyViewType();
}


void CameraView::setViewType(ViewType vt) {
    switch (vt) {
    case FRONT:orbit_angle_perspective=osg::Vec2(0,0);break;
    case BACK: orbit_angle_perspective=osg::Vec2(PI,0);break;
    case TOP:  orbit_angle_perspective=osg::Vec2(0,PI/2);break;
    case BOTTOM:orbit_angle_perspective=osg::Vec2(0,-PI/2);break;
    case LEFT: orbit_angle_perspective=osg::Vec2(-PI/2,0);break;
    case RIGHT:orbit_angle_perspective=osg::Vec2(PI/2,0);break;
    case USER:orbit_angle_perspective=osg::Vec2(PI/4,PI/4);break;
    }
    if (gridNode.valid()) {
	osg::Matrix matrix;
	switch (vt) {
	case FRONT:break;
	case BACK:  break;//matrix.makeRotate(ROT_90_RAD*2,1,0,0);break;
	case TOP:
	case BOTTOM:matrix.makeRotate(ROT_90_RAD,1,0,0);break;
	case LEFT:  
	case RIGHT: matrix.makeRotate(ROT_90_RAD,0,1,0);break;
	case USER:   matrix.makeRotate(ROT_90_RAD,1,0,0);break;
	}
	gridNode->setMatrix(matrix);
    }
    applyViewType();
}
