//------------------------------ $Keywords ----------------------------------
// SelfImage - Disk image manipulation program
// SelfImage_Utility.cpp - Utility and miscelaneous functions
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of SelfImage.
//
// SelfImageis free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// SelfImage is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 0 $ $Revision: 11 $
/*
$History: **** V 0.1 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-07-31 8:29:45 AM - 1963 Bytes
$History: * selfimage_utility.h - 2005-07-31 8:21:56 AM - 1395 Bytes
$History: * Initial check-in
$History: **** V 0.2 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-08-01 1:56:17 PM - 2833 Bytes
$History: * selfimage_utility.h - 2005-07-31 9:14:14 AM - 1439 Bytes
$History: * Add GetLastErrorMessage().
$History: **** V 0.3 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-08-01 3:11:51 PM - 3827 Bytes
$History: * selfimage_utility.h - 2005-08-01 2:13:58 PM - 1506 Bytes
$History: * Add MakeByteSizeLabel().
$History: **** V 0.4 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-08-06 6:29:09 PM - 13684 Bytes
$History: * selfimage_utility.h - 2005-08-06 6:13:00 PM - 6700 Bytes
$History: * Add support for unmounted volumes
$History: **** V 0.5 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-08-17 7:51:35 AM - 13878 Bytes
$History: * selfimage_utility.h - 2005-08-17 7:49:02 AM - 6635 Bytes
$History: * Change licensing - only version 2 of the GPL, no later versions
$History: **** V 0.6 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-11-07 2:41:16 AM - 9865 Bytes
$History: * selfimage_utility.h - 2005-11-02 10:42:08 PM - 6443 Bytes
$History: * Changes for 0.2 - too many to note
$History: **** V 0.7 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-11-10 10:05:01 AM - 15631 Bytes
$History: * selfimage_utility.h - 2005-11-10 9:34:10 AM - 6469 Bytes
$History: * Add version check
$History: **** V 0.8 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-11-12 4:41:53 PM - 15883 Bytes
$History: * selfimage_utility.h - 2005-11-10 9:34:10 AM - 6469 Bytes
$History: * Fix problem where version checks occur more than once a day
$History: **** V 0.9 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-11-12 5:22:12 PM - 16139 Bytes
$History: * selfimage_utility.h - 2005-11-12 5:20:06 PM - 6473 Bytes
$History: * Typo in copyright area + change in program description
$History: **** V 0.10 by kfitzner ****
$History: * selfimage_utility.cpp - 2005-11-14 3:51:59 PM - 16393 Bytes
$History: * selfimage_utility.h - 2005-11-12 5:20:06 PM - 6473 Bytes
$History: * Add label "bytes" to MakeByteLabel for small sizes
$History: **** Latest ** V 0.11 by kfitzner ** 2005-11-18 1:26:57 PM ****
$History: * Add GetFileSystemType() and GetDreeDiskSpace() functions
*/
//----------------------------  $NoKeywords ---------------------------------

//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#include <vcl.h>
#include <DateUtils.hpp>
#include <dir.h>
#include <IdBaseComponent.hpp>
#include <IdComponent.hpp>
#include <IdHTTP.hpp>
#include <IdTCPClient.hpp>
#include <IdTCPConnection.hpp>
#pragma hdrstop

#include "TConfiguration.h"
#include "TProgramLog.h"
#include "SelfImage_Exceptions.h"
#include "SelfImage_NewVersionNotify.h"
#include "SelfImage_Utility.h"
//---------------------------------------------------------------------------
extern TConfiguration *SelfImageConfig;
//---------------------------------------------------------------------------
static HMODULE ntdll = NULL;
static NtOpenDirectoryObject_t     NtOpenDirectoryObject     = NULL;
static NtQueryDirectoryObject_t    NtQueryDirectoryObject    = NULL;
static NtOpenSymbolicLinkObject_t  NtOpenSymbolicLinkObject  = NULL;
static NtQuerySymbolicLinkObject_t NtQuerySymbolicLinkObject = NULL;
static RtlInitUnicodeString_t      RtlInitUnicodeString      = NULL;
static NtCreateFile_t              NtCreateFile              = NULL;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall InitializeUtilityFunctions(void) {
  ntdll = GetModuleHandle("ntdll.dll");
  if (ntdll) {
    NtOpenDirectoryObject     = (NtOpenDirectoryObject_t)    GetProcAddress(ntdll, "NtOpenDirectoryObject");
    NtQueryDirectoryObject    = (NtQueryDirectoryObject_t)   GetProcAddress(ntdll, "NtQueryDirectoryObject");
    NtOpenSymbolicLinkObject  = (NtOpenSymbolicLinkObject_t) GetProcAddress(ntdll, "NtOpenSymbolicLinkObject");
    NtQuerySymbolicLinkObject = (NtQuerySymbolicLinkObject_t)GetProcAddress(ntdll, "NtQuerySymbolicLinkObject");
    NtCreateFile              = (NtCreateFile_t)             GetProcAddress(ntdll, "NtCreateFile");
    RtlInitUnicodeString      = (RtlInitUnicodeString_t)     GetProcAddress(ntdll, "RtlInitUnicodeString");
    HaveNTCalls = true;
  }  // if (ntdll)
}  // void __fastcall InitializeUtilityFunctions(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Get the last Windows error and format the message into an AnsiString
//
AnsiString __fastcall GetLastErrorMessage(void)
{
  char *buffer;
  AnsiString retval;

  FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, NULL, GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
    (LPTSTR) &buffer, 0, NULL );
  retval = String(buffer).Trim();
  LocalFree(buffer);

  return(retval);
}  // inline AnsiString GetLastErrorMessage(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Convert a number into a normalized string with a KB/MB/GB/TB suffix
// appropriate for a byte capacity label
//
AnsiString __fastcall MakeByteLabel(long double ddByteCount)
{
  AnsiString retval;

  if (ddByteCount > 1099511627776.0)
    retval.sprintf("%3.3LfTB", ddByteCount / 1099511627776.0);
  else if (ddByteCount > 1073741824.0)
    retval.sprintf("%3.3LfGB", ddByteCount / 1073741824.0);
  else if (ddByteCount > 1048576.0)
    retval.sprintf("%3.3LfMB", ddByteCount / 1048576.0);
  else if (ddByteCount > 1024.0)
    retval.sprintf("%3.3LfKB", ddByteCount / 1024.0);
  else retval.sprintf("%Lf Bytes",ddByteCount);

  return(retval);
}  // AnsiString __fastcall MakeByteSizeLabel(__int64 &nnByteCount)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Get the contents of an NT directory object - this uses some undocumented
// calls from ntdll.dll
//
TStringList * __fastcall GetNTDirectoryObjectContents(WideString Directory)
{
  TStringList *Entries;
  UNICODE_STRING usDir;
  OBJECT_ATTRIBUTES oa;
  HANDLE hDeviceDir;
  NTSTATUS nStatus;
  OBJDIR_INFORMATION *DirInfo;
  DWORD index;
  AnsiString Error;

  if (!HaveNTCalls)
    return NULL;

  RtlInitUnicodeString(&usDir, Directory);
  oa.Length = sizeof(OBJECT_ATTRIBUTES);
  oa.ObjectName = &usDir;
  oa.Attributes = OBJ_CASE_INSENSITIVE;
  oa.SecurityDescriptor = NULL;
  oa.SecurityQualityOfService = NULL;
  oa.RootDirectory = 0;
  nStatus = NtOpenDirectoryObject(&hDeviceDir, STANDARD_RIGHTS_READ | DIRECTORY_QUERY, &oa);
  if (!NT_SUCCESS(nStatus))
    return NULL;
  DirInfo = (OBJDIR_INFORMATION *)malloc(2048);
  index = 0;
  Entries = new TStringList;
  while (NT_SUCCESS(NtQueryDirectoryObject(hDeviceDir, DirInfo, 1024, true, false, &index, NULL)))
    Entries->Add(Directory + "\\" + String(DirInfo->ObjectName.Buffer));
  CloseHandle(hDeviceDir);
  free(DirInfo);
  if (Entries->Count)
    return Entries;
  else {  // if (!Entries->Count)
    delete Entries;
    return NULL;
  }  // if (!Entries->Count)
}   // TStringList * __fastcall GetNTDirectoryObjectContents(WideString Directory)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
AnsiString __fastcall GetNTLinkDestination(WideString Source)
{
  UNICODE_STRING usName;
  OBJECT_ATTRIBUTES oa;
  NTSTATUS nStatus;
  HANDLE hLink;
  wchar_t *buffer;
  AnsiString retval;

  RtlInitUnicodeString(&usName, Source);
  oa.Length = sizeof(OBJECT_ATTRIBUTES);
  oa.RootDirectory = 0;
  oa.ObjectName = &usName;
  oa.Attributes = OBJ_CASE_INSENSITIVE;
  oa.SecurityDescriptor = NULL;
  oa.SecurityQualityOfService = NULL;
  nStatus = NtOpenSymbolicLinkObject(&hLink, SYMBOLIC_LINK_QUERY, &oa);
  if (NT_SUCCESS(nStatus)) {
    buffer = (wchar_t *)malloc(2048);
    usName.Length = 0;
    usName.MaximumLength = 1024;
    usName.Buffer = buffer;
    nStatus = NtQuerySymbolicLinkObject(hLink, &usName, NULL);
    usName.Buffer[usName.Length/2] = 0;
    if (NT_SUCCESS(nStatus))
      retval = String(usName.Buffer);
    free(buffer);
  }  // if (NTSUCCESS(nStatus))
  return retval;
}  // AnsiString __fastcall GetNTLinkDestination(AnsiString Source)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// This is a wrapper function around the native NtCreateFile() function.
// Officially, NtCreateFile() is undocumented, but it is widely used.
//
HANDLE NTOpen(WideString FileName, ACCESS_MASK DesiredAccess, ULONG FileAttributes, ULONG ShareAccess, ULONG CreateDisposition, ULONG CreateOptions)
{
  UNICODE_STRING usFileName;
  OBJECT_ATTRIBUTES oa;
  HANDLE hFile;
  IO_STATUS_BLOCK ios;
  NTSTATUS nStatus;

  if (!HaveNTCalls)
    return INVALID_HANDLE_VALUE;
  RtlInitUnicodeString(&usFileName, FileName);
  oa.Length = sizeof(oa);
  oa.RootDirectory = 0;
  oa.ObjectName = &usFileName;
  oa.Attributes = OBJ_CASE_INSENSITIVE;
  oa.SecurityDescriptor = NULL;
  oa.SecurityQualityOfService = NULL;
  nStatus = NtCreateFile(&hFile, DesiredAccess, &oa, &ios, NULL, FileAttributes, ShareAccess, CreateDisposition, CreateOptions, NULL, 0);
  if (NT_SUCCESS(nStatus))
    return hFile;
  else
    return INVALID_HANDLE_VALUE;
}  // HANDLE XpCreateFile()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Check to see if SelfImage is up-to-date - notify the user if it isn't
//
void CheckVersion(void) {

  TDateTime *dtLastCheck;
  char *filename = NULL;
  char *buffer = NULL;
  char *verbuff = NULL;
  unsigned nVersionInfoSize;
  int nBuildPos;
  static AnsiString sMyVersion = "";
  static int nMyBuild = 0;
  AnsiString sCurrentVersion, sURL;
  int nCurrentBuild = 0;
  TIdHTTP *HTTP;
  DWORD nDummy;

  __ENTERFUNCTION__;

  try {
    try {
      formSelfImageNewVersionNotify = NULL;
      if (!SelfImageConfig->Values["Update Notify"]) {
        LOG(LOG_DEBUG, "Update Notify is turned off - not checking for new version.");
        __RETURNFUNCTION__;
      }  // if (!SelfImageConfig->Values["Update Notify"])

      if (String(SelfImageConfig->Values["Last Version Check"]).IsEmpty())
        SelfImageConfig->Values["Last Version Check"] = (Now()-1).DateTimeString();
      dtLastCheck = new TDateTime(String(SelfImageConfig->Values["Last Version Check"]));
      if (!DaysBetween(Now(), *dtLastCheck)) {
        LOG1(LOG_DEBUG, "Last new version check was performed %s - not checking again.", dtLastCheck->DateTimeString().c_str());
        __RETURNFUNCTION__;
      }  // if (!DaysBetween(Now(), *dtLastCheck))

      SelfImageConfig->Values["Last Version Check"] = Now().DateTimeString();

      // Get this executable's version - this is a pain, so let's just do it once for each time SelfImage is instantiated
      if (!nMyBuild) {
        LOG(LOG_DEBUG, "Retrieving internal version number.");
        filename = (char *)malloc(MAXPATH);
        GetModuleFileName(HInstance, filename, MAXPATH);
        nVersionInfoSize = GetFileVersionInfoSize(filename, &nDummy);
        // These following three exceptions should never make it to the user except through the logs in a debug build.
        // Since they are only for logging, they are allowed to have text that doesn't come from the string table.
        if (!nVersionInfoSize)
          throw ESelfImageSystemError("GetFileVersionInfoSize() failed: " + GetLastErrorMessage());
        buffer = (char *)malloc(nVersionInfoSize);
        if (!GetFileVersionInfo(filename, NULL, nVersionInfoSize, buffer))
          throw ESelfImageSystemError("GetFileVersionInfo() failed:" + GetLastErrorMessage());
        if (!VerQueryValue(buffer, "\\StringFileInfo\\100904E4\\FileVersion", (void **)&verbuff, NULL))
          throw ESelfImageSystemError("VerQueryValue() failed:" + GetLastErrorMessage());
        sMyVersion = verbuff;
        nBuildPos = sMyVersion.LastDelimiter(".");
        nMyBuild = sMyVersion.SubString(nBuildPos + 1, sMyVersion.Length() - nBuildPos).ToInt();
        sMyVersion = sMyVersion.SubString(1,nBuildPos-1) + " (Build " + String(nMyBuild) + ")";
        LOG1(LOG_MESSAGE, "Internal version string retrieved from resources: \"%s\".", sMyVersion.c_str());
      }  // if (sMyVersion.IsEmpty())

      // Get SelfImage's current version from the interweb - woo, this is cool
      HTTP = new TIdHTTP(NULL);
      sURL = SelfImageConfig->Values["Version URL"];
      LOG1(LOG_DEBUG, "Attempting to retrieve version string from URL \"%s\".", sURL.c_str());
      try {
        sCurrentVersion = HTTP->Get(SelfImageConfig->Values["Version URL"]);
        nBuildPos = sCurrentVersion.Pos("(Build ");
        nCurrentBuild = sCurrentVersion.SubString(nBuildPos + 7, sCurrentVersion.Length() - nBuildPos - 7).ToInt();
      }
      catch (...) { sCurrentVersion = ""; }
      delete HTTP;
      #ifdef __LOGGINGENABLED__
      if (!sCurrentVersion.IsEmpty())
        LOG2(LOG_MESSAGE, "Version string \"%s\" retrieved from %s.", sCurrentVersion.c_str(), sURL.c_str())
      else
        LOG2(LOG_WARNING, "Unable to retrieve version string from URL \"%s\": %s", sURL.c_str(), HTTP->ResponseText.c_str());
      #endif

      // Compare the two...
      if (nCurrentBuild && nMyBuild < nCurrentBuild && sCurrentVersion != String(SelfImageConfig->Values["Notify Squelch"])) {
        LOG(LOG_DEBUG, "Notifying user of new version.");
        formSelfImageNewVersionNotify = new TformSelfImageNewVersionNotify(NULL, sMyVersion, sCurrentVersion);
        if (formSelfImageNewVersionNotify->ShowModal() == mrCancel)
          SelfImageConfig->Values["Notify Squelch"] = sCurrentVersion;
      }  // if (different versions)
    }  // inner try
    catch (Exception &e) {
      LOG2(LOG_ERROR, "Exception of type \"%s\" raised during new version check with message \"%s\"", String(e.ClassName()).c_str(), e.Message.c_str());
    }  // catch
  }  // outer try
  __finally {
    if (filename)
      free(filename);
    if (buffer)
      free(buffer);
    if (formSelfImageNewVersionNotify) {
      delete formSelfImageNewVersionNotify;
      formSelfImageNewVersionNotify = NULL;
    }  // if (formSelfImageNewVersionNotify)
  }
  __RETURNFUNCTION__;
}  // void CheckVersion(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Returns the type of filesystem that a file resides on.
//
AnsiString GetFileSystemType(AnsiString FileName)
{
  AnsiString retval;
  char *VolumeName, *FileSystemName;
  DWORD VolumeSerialNumber, MaxComponentLength, FileSystemFlags;
  bool success;

  VolumeName = new char[MAX_PATH];
  FileSystemName = new char[MAX_PATH];
  success = GetVolumeInformation((ExtractFileDrive(FileName) + "\\").c_str(), VolumeName, MAX_PATH, &VolumeSerialNumber, &MaxComponentLength, &FileSystemFlags, FileSystemName, 64);
  if (success)
    retval = String(FileSystemName);
  delete VolumeName;
  delete FileSystemName;

  return String(FileSystemName);
}  // AnsiString GetFileSystemType(AnsiString FileName)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Return the amount of free space on the volume that a file resides on.
//
__int64 GetFreeDiskSpace(AnsiString FileName)
{
  unsigned __int64 FreeBytesAvailable, TotalBytes, TotalFreeBytes;
  bool success;

  success = ::GetDiskFreeSpaceEx((ExtractFileDrive(FileName) + "\\").c_str(), (ULARGE_INTEGER *)&FreeBytesAvailable, (ULARGE_INTEGER *)&TotalBytes, (ULARGE_INTEGER *)&TotalFreeBytes);
  if (success)
    return FreeBytesAvailable;
  else
    return 0;
}  // __int64 GetFreeSpace(AnsiString FileName)
//---------------------------------------------------------------------------

