//------------------------------ $Keywords ----------------------------------
// SelfImage - Disk image manipulation program
// SelfImage_TImageStore.cpp - TImageStore class
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of SelfImage.
//
// SelfImage is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// SelfImage is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 0 $ $Revision: 2 $
/*
$History: **** V 0.1 by kfitzner ****
$History: * selfimage_timagestore.cpp - 2005-11-07 2:39:39 AM - 8573 Bytes
$History: * selfimage_timagestore.h - 2005-10-21 10:56:46 AM - 2239 Bytes
$History: * Initial check-in
$History: **** Latest ** V 0.2 by kfitzner ** 2005-11-12 5:22:11 PM ****
$History: * Typo in copyright area + change in program description
*/
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 12 Sep 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// An image "store" represtents any place where, no surprise, an image can
// be stored or retrieved from.  A file, a partition, even a network source.
// This class abstracts opens, closes, reads, etc so that the other classes
// don't need to know (or care) where their data is coming from or going.
//
// This class doesn't need to be particularly thread-safe, since it's a
// little unlikely that, say, the read thread is going to want to try and
// access the write store.
//---------------------------------------------------------------------------
#include <vcl.h>
#include <stdio.h>
#pragma hdrstop

#include "SelfImage_Utility.h"
#include "SelfImage_Exceptions.h"
#include "SelfImage_TImageStore.h"
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TImageStore constructor that opens the target at the same time
//
__fastcall TImageStore::TImageStore(TImageStoreType isType, AnsiString Target, bool WriteAccess) {
  FType = isType;
  FFileName = Target;
  bWriteAccess = WriteAccess;
  if (!Open())
    throw ESelfImageError("Could not open store target \"" + Target + "\": " + GetLastErrorMessage());
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TImageStore::~TImageStore()
//
__fastcall TImageStore::~TImageStore() {
  Close();
}  // __fastcall TImageStore::~TImageStore()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Open an image store - this determines the type of the store and whether
// or not NT open functions are available and then decides the best way
// to open the target.
//
bool __fastcall TImageStore::Open(void) {

  switch (Type) {
    case isFile: {
      LARGE_INTEGER nnFileSize;

      DWORD Access     = GENERIC_READ | (bWriteAccess?GENERIC_WRITE:0);
      DWORD ShareMode  = (bWriteAccess?0:FILE_SHARE_READ);
      DWORD CreateMode = (bWriteAccess?CREATE_ALWAYS:OPEN_EXISTING);
      hHandle = CreateFile(FileName.c_str(), Access, ShareMode, NULL, CreateMode, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hHandle != INVALID_HANDLE_VALUE) {
        if (!bWriteAccess) {
          nnFileSize.LowPart = GetFileSize(hHandle, (unsigned long *)&nnFileSize.HighPart);
          FGeometry.Bytes = nnFileSize.QuadPart;
        } else
          FGeometry.Bytes = -1;
      }  // if (hHandle != INVALID_HANDLE_VALUE)
      break;
    }  // case isFile:
    case isDrive: {
      PARTITION_INFORMATION PartitionInfo;
      DISK_GEOMETRY DiskGeometry;
      DWORD Dummy;

      DWORD Access     = GENERIC_READ | (bWriteAccess?GENERIC_WRITE:0);
      DWORD ShareMode  = (bWriteAccess?0:FILE_SHARE_READ|FILE_SHARE_WRITE);
      if (HaveNTCalls)
        hHandle = NTOpen(FileName, SYNCHRONIZE | Access, FILE_ATTRIBUTE_NORMAL, ShareMode, FILE_OPEN, FILE_SYNCHRONOUS_IO_NONALERT | FILE_SEQUENTIAL_ONLY);
      else
        hHandle = CreateFile(FileName.c_str(), GENERIC_READ, ShareMode, NULL, OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN | FILE_FLAG_BACKUP_SEMANTICS, NULL);
      if (hHandle != INVALID_HANDLE_VALUE) {
        DeviceIoControl(hHandle, IOCTL_DISK_GET_PARTITION_INFO, NULL, 0, &PartitionInfo, sizeof(PartitionInfo), &Dummy, NULL);
        DeviceIoControl(hHandle, IOCTL_DISK_GET_DRIVE_GEOMETRY	, NULL, 0, &DiskGeometry, sizeof(DiskGeometry), &Dummy, NULL);
        if (FileName.Pos("Floppy") || FileName.Pos("A:") || FileName.Pos("B:"))
          FGeometry.Bytes = DiskGeometry.Cylinders.QuadPart * DiskGeometry.TracksPerCylinder * DiskGeometry.SectorsPerTrack * DiskGeometry.BytesPerSector;
        else
          FGeometry.Bytes = PartitionInfo.PartitionLength.QuadPart;
        FGeometry.BytesPerSector = DiskGeometry.BytesPerSector;
        FGeometry.Sectors = Geometry.Bytes / Geometry.BytesPerSector;
      }  // if (hHandle != INVALID_HANDLE_VALUE)
      break;
    }  // case isDrive:
  }  // switch (Type)

  return (hHandle != INVALID_HANDLE_VALUE);
}  // bool __fastcall TImageStore::Open(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Close() - Close an image store
//
bool __fastcall TImageStore::Close(void) {
  return CloseHandle(hHandle);
}  // bool __fastcall Close(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Read() - As the name suggests, read from the file/device.  Special care
// has to be taken when reading from handles created with "NtCreateFile()".
// Windows versions based on NT have problems with partial reads of disk
// devices opened with NtCreateFile().  When there isn't enough data in the
// file to satisfy the entire length of the read, ReadFile() is supposed to
// always read as many as there are.  Disk devices of the type
// "\Device\Harddisk0\Partition1" opened with NtCreateFile don't do this.
// When there isn't enough data left to satisfy the entire read, ReadFile()
// will fail entirely and GetLastError() will return ERROR_INVALID_PARAMETER.
// To compensate for this, whenever we encounter ERROR_INVALID_PARAMETER, we
// step down the size of the read to the device's sector size and then read
// until we get an error again.
//
bool __fastcall TImageStore::Read(void *buffer, int nLength, unsigned *nBytesRead) {
  unsigned nBytesReadTemp, pos;
  bool bSuccess;
  bSuccess = ReadFile(hHandle, buffer, nLength, (unsigned long *)&nBytesReadTemp, NULL);
  if (!bSuccess && GetLastError() == ERROR_INVALID_PARAMETER) {
    // ERROR_INVALID_PARAMETER - step down read size as discussed above
    pos = 0;
    while (ReadFile(hHandle, (char *)buffer + pos, Geometry.BytesPerSector?Geometry.BytesPerSector:512, (unsigned long *)&nBytesReadTemp, NULL))
      pos += nBytesReadTemp;
    nBytesReadTemp = pos;
    bSuccess = true;
  }  // if (!bSuccess)
  *nBytesRead = nBytesReadTemp;
  return bSuccess;
}  // bool __fastcall TImageStore::Read(void *buffer, int nLength, unsigned *nBytesRead)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Write() - similar to Read() but different.
//
bool __fastcall TImageStore::Write(void *buffer, int nLength, unsigned *nBytesWritten) {
  return (WriteFile(hHandle, buffer, nLength, (unsigned long *)nBytesWritten, NULL));
}  // bool __fastcall TImageStore::Write(void *buffer, int nLength, unsigned *nBytesWritten)
//---------------------------------------------------------------------------

