//------------------------------ $Keywords ----------------------------------
// SelfImage - Disk image manipulation program
// SelfImage_Buffer.cpp - TImageBuffer class
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of SelfImage.
//
// SelfImage is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// SelfImage is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 0 $ $Revision: 2 $
/*
$History: **** V 0.1 by kfitzner ****
$History: * selfimage_timagebuffer.cpp - 2005-11-07 2:39:38 AM - 11089 Bytes
$History: * selfimage_timagebuffer.h - 2005-11-07 2:39:38 AM - 3741 Bytes
$History: * Initial check-in
$History: **** Latest ** V 0.2 by kfitzner ** 2005-11-12 5:22:11 PM ****
$History: * Typo in copyright area + change in program description
*/
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 13 Sep 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// The TImageBuffer class is used to hold and arbitrate allocation of a group
// memory of "chunks" used for buffering.  A circular buffer is used to hold
// pointers to the buffer chunks.  These chunks are allocated off the bottom
// for writing and off the top for reading.
//
// Reading and writing, as referred to in this unit, are relative to THIS
// class.  Writing means writing TO a buffer chunk - reading is reading from
// it.  Thus, a thread that is reading data will allocate a write chunk as a
// place to write its data to.
//
// Care needs to be taken here, as this class is a memory buffer arbitration
// class for multiple threads.  Thread safety is important.  An instance of
// TMultiReadExclusiveWriteSynchronizer (that's a mouthfull!) is used instead
// of critical sections.  Most of the time there will be no buffers free to
// allocate and the allocation method will block.  In this case, since we're
// not writing, there is no need to completely linearize the method with a
// critical section.  The multi-read synchronizer lets us get away with this.
//
// Care also needs to be taken about the order that chunks are allocated out.
// Once multi-threaded compression is implemented, this becomes possible:
//   - Compression thread #1 allocates chunk 1 for writing to
//   - Compression thread #2 allocates chunk 2 for writing to
//   - Compression thread #2 completes chunk 2 and releases it
//   - Compression thread #1 completes chunk 1 and releases it
// We could put serial numbers inside the chunks, but then we would have to
// deal with reordering them.  Instead, we guarantee that chunks will be
// allocated out in the same order every time.  The same order they are
// allocated out for writing will be the same order they are allocated out
// for reading.  Now it doesn't matter if a 'later' block is processed
// before an 'earlier' block - they will be served out in the correct order.
//---------------------------------------------------------------------------
#include <vcl.h>
#include <Math.hpp>
#pragma hdrstop

#include "SelfImage_Exceptions.h"
#include "SelfImage_Utility.h"
#include "SelfImage_TImageBuffer.h"
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TBufferChunk class
//

//---------------------------------------------------------------------------
// TBufferChunk constructor
//
TBufferChunk::TBufferChunk(int nSize, int nIndex) {

  FData = VirtualAlloc(NULL, nSize, MEM_COMMIT, PAGE_READWRITE);
  if (!FData)
    throw ESelfImageAllocError("SelfImage failed to allocate " + String(nSize) + " bytes of memory: " + GetLastErrorMessage());

  FMaxSize = nSize;
  FUsedSize = SIZE_NOT_SET;
  FEOF = false;
  FIndex = nIndex;
}  // TBufferChunk::TBufferChunk(int nSize, int nIndex)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TBufferChunk destructor
//
TBufferChunk::~TBufferChunk() {
  VirtualFree(FData, 0, MEM_RELEASE);
}  // TBufferChunk::~TBufferChunk()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TImageBuffer class
//

//---------------------------------------------------------------------------
// TImageBuffer constructor
//
__fastcall TImageBuffer::TImageBuffer(int Size, int Count) {

  Sync = new TMultiReadExclusiveWriteSynchronizer();

  ChunkSize = Size;
  ChunkCount = 1 << (int)Ceil(Log2(Count));
  IndexMask = ChunkCount - 1;
  Chunks = new TBufferChunk *[ChunkCount];
  ChunkStates = new TBufferChunkState [ChunkCount];

  NextReadChunkIndex = NextWriteChunkIndex = 0;

  // Create all the buffer chunks
  for (unsigned n = 0; n < ChunkCount; n++) {
    Chunks[n] = new TBufferChunk(ChunkSize, n);
    ChunkStates[n] = csFree;
  }  // for (unsigned n = 0; n < nChunkCount; n++)

  // Initialize our event objects
  ReadChunkReady  = new TEvent(NULL, true, false, "Read Chunk Ready");
  WriteChunkReady = new TEvent(NULL, true, false, "Write Chunk Ready");

}  // TImageBuffer::TImageBuffer(int nChunkSize, int nChunkCount)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Allocate a chunk.  This has to be thread safe!
//
TBufferChunk * __fastcall TImageBuffer::GetChunk(TBufferChunkState Purpose, bool *TerminationFlag) {
  unsigned *pnIndexMarker;
  TBufferChunkState DesiredInitialChunkState, NewChunkState;
  TEvent *WaitObject;
  bool bChunkObtained = false;
  TBufferChunk *retval = NULL;
  #ifdef _DEBUG
  int nRaceCounter = 1000;
  #endif

  // Set up the function for either read or write chunk allocation
  if (Purpose == csAllocRead) {
    pnIndexMarker = &NextReadChunkIndex;
    DesiredInitialChunkState = csFinishedWrite;
    NewChunkState = csAllocRead;
    WaitObject = ReadChunkReady;
  } else if (Purpose == csAllocWrite) {
    pnIndexMarker = &NextWriteChunkIndex;
    DesiredInitialChunkState = csFree;
    NewChunkState = csAllocWrite;
    WaitObject = WriteChunkReady;
  }  // if (Purpose == ...)

  // Loop until we get an allocation
  while (!bChunkObtained) {
    Sync->BeginRead();
    if (ChunkStates[*pnIndexMarker] != DesiredInitialChunkState) {
      // No free chunks - wait
      Sync->EndRead();
      WaitObject->WaitFor(1000);
      // Check to see if the calling thread is being told to terminate.  If so, then return back.
      if (*TerminationFlag)
        return(NULL);
    }  // if (nNewBottomAlloc == nTopChunk)
    else {  // if (ChunkStatus[*IndexMarker] == DesiredInitialChunkState)
      // There IS a free chunk, so get a write lock
      Sync->EndRead();
      Sync->BeginWrite();
      // Another thread could have been waiting (eventually with compression there will be multiple reader/writer threads)
      // so even though we checked, now we have the write lock we have to check again to ensure that some other thread didn't
      // snatch the chunk before we got here.
      if (ChunkStates[*pnIndexMarker] == DesiredInitialChunkState) {
        // Free chunk is still there and we have a write lock, so we're good to go - allocate it
        retval = Chunks[*pnIndexMarker];
        ChunkStates[*pnIndexMarker] = NewChunkState;
        *pnIndexMarker = (*pnIndexMarker + 1) & IndexMask;
        // If the next chunk in the ring isn't free then it means we're full, so make sure the wait event is reset
        if (ChunkStates[*pnIndexMarker] != DesiredInitialChunkState)
          WaitObject->ResetEvent();
        // Set the flag stating we have our chunk
        bChunkObtained = true;
      }  // if (ChunkStatus[IndexAllocatedTop] == caFree)
      Sync->EndWrite();
    }  // else (ChunkStatus[IndexAllocatedBottom] != caFree)

    #ifdef _DEBUG
    // Check for a race or some other bug - if we've looped a thousand times with no result, there's a problem
    if (!nRaceCounter--)
      throw ESelfImageAllocError("Buffer chunk allocation failed - possible race condition.");
    #endif

    // If we are looping, it's probably because some other thread has allocated the chunk we wanted out from
    // under us, but hasn't reset WaitObject (ReadChunkReady or WriteChunkReady) yet - so let's give up the
    // rest of this thread's timeslice before we loop so that the other thread can finish up.
    if (!bChunkObtained)
      Sleep(0);
  }  // while (!bChunkObtained)

  return retval;
}  // TBufferChunk *TImageBuffer::GetChunk(TBufferChunkAlloc Purpose)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Release a chunk
//
bool __fastcall TImageBuffer::ReleaseChunk(TBufferChunk *Chunk) {
  unsigned nChunkIndex;
  unsigned nMarkerIndex;
  TBufferChunkState StateAfterRelease;
  TEvent *WaitObject;

  nChunkIndex = Chunk->Index;
  // We're going to have to perform an update, so get a write lock right from the beginning - saves relocking
  Sync->BeginWrite();

  // Configure the function for either releasing a chunk previously allocated for reading or writing
  if (ChunkStates[nChunkIndex] == csAllocRead) {
    StateAfterRelease = csFree;
    nMarkerIndex = NextWriteChunkIndex;
    WaitObject = WriteChunkReady;
    // This chunk is now free for reuse, so reset it
    Chunk->Reset();
  } else if (ChunkStates[nChunkIndex] == csAllocWrite) {
    StateAfterRelease = csFinishedWrite;
    nMarkerIndex = NextReadChunkIndex;
    WaitObject = ReadChunkReady;
  }  // if (ChunkStates[nIndex] = ...)

  ChunkStates[nChunkIndex] = StateAfterRelease;
  // If we've just released the chunk that is pointed to by the marker (the marker that is relevant to the state that
  // the chunk is now), then we signal correct event in case some other thread is waiting for it.
  if (nChunkIndex == nMarkerIndex)
    WaitObject->SetEvent();
  Sync->EndWrite();

  return true;
}  // bool __fastcall ReleaseChunk(TBufferChunk *Chunk)
//---------------------------------------------------------------------------

