//------------------------------ $Keywords ----------------------------------
// SelfImage - Disk image manipulation program
// SelfImage_ReadThread.cpp - Data read thread class
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of SelfImage.
//
// SelfImage is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// SelfImage is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 0 $ $Revision: 3 $
/*
$History: **** V 0.1 by kfitzner ****
$History: * selfimage_readthread.cpp - 2005-11-07 2:39:38 AM - 6668 Bytes
$History: * selfimage_readthread.h - 2005-11-07 1:48:52 AM - 2377 Bytes
$History: * Initial check-in
$History: **** V 0.2 by kfitzner ****
$History: * selfimage_readthread.cpp - 2005-11-12 5:22:11 PM - 6929 Bytes
$History: * selfimage_readthread.h - 2005-11-12 5:18:28 PM - 2381 Bytes
$History: * Typo in copyright area + change in program description
$History: **** Latest ** V 0.3 by kfitzner ** 2005-11-18 3:30:51 PM ****
$History: * Fix bug where exception thrown when read input ends 
$History: * on a buffer-size boundary
*/
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 8 Sep 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// About to embark on a whole new structure for SelfImage.  Reading and
// writing are to be separated in their own threads and will be abstracted
// through the TImageBuffer class.  This read thread (read from disk/file)
// is given a buffer to write its data to.  The write thread will be given a
// buffer to read from.  This will eventually allow sticking threads in the
// middle to process the data too.  The intention with that is to allow
// compression threads (compresseion thread reads from one buffer,
// compresses, writes to another).  This will allow chaining of the threads
// together.
//
// Borland's thread object wizard dumps the following note in thread object
// units that it creates:
//
//   Important: Methods and properties of objects in VCL can only be
//   used in a method called using Synchronize, for example:
//
//      Synchronize(UpdateCaption);
//
//   where UpdateCaption could look like:
//
//      void __fastcall TSelfImageReadThread::UpdateCaption()
//      {
//        Form1->Caption = "Updated in a thread";
//      }
//
// Only one thread at a time can use VCL - ug, what a horrible design. :p
// Ok, that means for piping data from the read/write threads back to the
// user, we want to update memory variables only and let the main thread
// actually do the UI updating.  This way we won't impact performance by
// strangling the read or write threads with linearized VCL calls.
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "SelfImage_ReadThread.h"
#include "SelfImage_Utility.h"
#include "SelfImage_Exceptions.h"
#pragma package(smart_init)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Construction zone ahead - this is a hardhat area.
//
__fastcall TSelfImageReadThread::TSelfImageReadThread(TImageStore *ReadStore, TImageBuffer *ImageBuffer): TThread(true)
{
  this->ReadStore   = ReadStore;
  this->ImageBuffer = ImageBuffer;
  BytesRead = 0;
  FFinished = false;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Function to set the thread's name for debugging purposes.  This is done
// by filling in a special structure and firing off an exception.  A
// compatible debugger knows to look in exceptions of this type and look for
// the thread's name.  A little bit of an odd way of doing things, but it
// seems to work.
//
void TSelfImageReadThread::SetName()
{
  THREADNAME_INFO info;
  info.dwType = 0x1000;
  info.szName = "ReadThread";
  info.dwThreadID = -1;
  info.dwFlags = 0;

  __try {
    RaiseException( 0x406D1388, 0, sizeof(info)/sizeof(DWORD),(DWORD*)&info );
  }
  __except (EXCEPTION_CONTINUE_EXECUTION) {}
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Off with the thread's head!  Execute time - read the image.
// Allocate a chunk, read from the store into the chunk, and then release
// the chunk.  This is the place to keep things very simple.
//
void __fastcall TSelfImageReadThread::Execute()
{
  bool bEOF = false;
  bool bReadSuccess;
  unsigned nBytesRead;

  SetName();

  try {
    while (!bEOF && !Terminated) {
      TBufferChunk *WriteChunk = ImageBuffer->GetChunk(csAllocWrite, &Terminated);
      if (!WriteChunk)
        continue;
      bReadSuccess = ReadStore->Read(WriteChunk->Data, WriteChunk->Size, &nBytesRead);
      // If we didn't get as much data as we expected, then we're at the end of the file.  Set the EOF marker
      // in the buffer chunk so the write thread knows this is the last.
      if (bReadSuccess && WriteChunk->Size != nBytesRead) {
        WriteChunk->Size = nBytesRead;
        WriteChunk->EOF = true;
        bEOF = true;
      }  // if (nBytesRead && WriteChunk->Size != nBytesRead)
      else if (!bReadSuccess) {
        // An error occured while reading - handle this.
        throw ESelfImageFileError("Error while reading from " + ReadStore->FileName + ": " + GetLastErrorMessage());
      }  // else if (!nBytesRead)
      BytesRead += nBytesRead;
      ImageBuffer->ReleaseChunk(WriteChunk);
    }  // while (!EOF && !Terminated)
    FFinished = true;
  }  // try
  // C++ Builder doesn't support passing exceptions from a thread.  So, we have to catch the exception here and
  // manually pass word that an error has occured.  This is done with the ErrorFlag property.
  catch (Exception &e) {
    FErrorFlag = true;
    FErrorMessage = "Read thread encountered " + e.ClassName() + " exception: \"" + e.Message + "\"";
    FFinished = true;
  }  // catch
}  // void __fastcall TSelfImageReadThread::Execute()
//---------------------------------------------------------------------------
