//------------------------------ $Keywords ----------------------------------
// SelfImage - Disk image manipulation program
// SelfImage_MainForm.cpp - Main form
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of SelfImage.
//
// SelfImage is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// SelfImage is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 0 $ $Revision: 15 $
/*
$History: **** V 0.1 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-07-30 6:44:58 PM - 2973 Bytes
$History: * selfimage_mainform.dfm - 2005-07-30 5:52:20 PM - 3054 Bytes
$History: * selfimage_mainform.h - 2005-07-30 6:44:58 PM - 1495 Bytes
$History: * selfimage_mainform.ddp - 2005-07-30 6:44:56 PM - 51 Bytes
$History: * Initial check-in
$History: **** V 0.2 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-08-01 3:11:59 PM - 11269 Bytes
$History: * selfimage_mainform.dfm - 2005-08-01 2:48:30 PM - 6098 Bytes
$History: * selfimage_mainform.h - 2005-08-01 2:38:08 PM - 3569 Bytes
$History: * selfimage_mainform.ddp - 2005-08-01 3:09:38 PM - 51 Bytes
$History: * Initial coding.
$History: **** V 0.3 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-08-03 4:54:43 AM - 13643 Bytes
$History: * selfimage_mainform.dfm - 2005-08-01 2:48:30 PM - 6098 Bytes
$History: * selfimage_mainform.h - 2005-08-02 9:33:32 PM - 3585 Bytes
$History: * selfimage_mainform.ddp - 2005-08-02 10:21:38 PM - 51 Bytes
$History: * Change writes to use asynchronous I/O
$History: **** V 0.4 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-08-06 6:29:05 PM - 17290 Bytes
$History: * selfimage_mainform.dfm - 2005-08-06 5:44:58 PM - 6762 Bytes
$History: * selfimage_mainform.h - 2005-08-06 4:39:32 PM - 3586 Bytes
$History: * selfimage_mainform.ddp - 2005-08-06 6:26:46 PM - 51 Bytes
$History: * Add support for unmounted volumes
$History: **** V 0.5 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-08-07 1:43:29 AM - 17659 Bytes
$History: * selfimage_mainform.dfm - 2005-08-06 6:45:06 PM - 6767 Bytes
$History: * selfimage_mainform.h - 2005-08-06 4:39:32 PM - 3586 Bytes
$History: * selfimage_mainform.ddp - 2005-08-06 6:48:38 PM - 51 Bytes
$History: * Add version to form caption
$History: **** V 0.6 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-08-17 7:51:34 AM - 17999 Bytes
$History: * selfimage_mainform.dfm - 2005-08-06 6:45:06 PM - 6767 Bytes
$History: * selfimage_mainform.h - 2005-08-17 7:48:54 AM - 3521 Bytes
$History: * selfimage_mainform.ddp - 2005-08-17 7:51:34 AM - 51 Bytes
$History: * Change licensing - only version 2 of the GPL, no later versions
$History: **** V 0.7 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-09-04 1:22:56 AM - 18470 Bytes
$History: * selfimage_mainform.dfm - 2005-08-06 6:45:06 PM - 6767 Bytes
$History: * selfimage_mainform.h - 2005-08-17 7:48:54 AM - 3521 Bytes
$History: * selfimage_mainform.ddp - 2005-09-04 1:22:40 AM - 51 Bytes
$History: * Fix variable not used warning - this probably won't 
$History: * matter thought because of the new threading structure coming in
$History: **** V 0.8 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-07 2:41:23 AM - 23870 Bytes
$History: * selfimage_mainform.dfm - 2005-11-07 1:42:56 AM - 8194 Bytes
$History: * selfimage_mainform.h - 2005-11-06 6:14:52 AM - 4034 Bytes
$History: * selfimage_mainform.ddp - 2005-11-07 2:41:22 AM - 51 Bytes
$History: * Changes for 0.2 - too many to note
$History: **** V 0.9 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-07 3:30:09 AM - 24240 Bytes
$History: * selfimage_mainform.dfm - 2005-11-07 3:28:12 AM - 8203 Bytes
$History: * selfimage_mainform.h - 2005-11-06 6:14:52 AM - 4034 Bytes
$History: * selfimage_mainform.ddp - 2005-11-07 3:28:28 AM - 51 Bytes
$History: * Add BETA tag to form caption
$History: **** V 0.10 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-15 5:30:11 PM - 32896 Bytes
$History: * selfimage_mainform.dfm - 2005-11-15 1:41:54 PM - 10185 Bytes
$History: * selfimage_mainform.h - 2005-11-15 1:41:54 PM - 4817 Bytes
$History: * selfimage_mainform.ddp - 2005-11-15 1:41:52 PM - 51 Bytes
$History: * Add menus, help button, and main form positioning. 
$History: *  Also added sanity check warnings.
$History: **** V 0.11 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-18 12:13:29 PM - 33862 Bytes
$History: * selfimage_mainform.dfm - 2005-11-16 8:30:30 AM - 10294 Bytes
$History: * selfimage_mainform.h - 2005-11-15 8:37:44 PM - 4890 Bytes
$History: * selfimage_mainform.ddp - 2005-11-18 12:13:28 PM - 51 Bytes
$History: * Add about box
$History: **** V 0.12 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-18 1:27:40 PM - 35241 Bytes
$History: * selfimage_mainform.dfm - 2005-11-16 8:30:30 AM - 10294 Bytes
$History: * selfimage_mainform.h - 2005-11-15 8:37:44 PM - 4890 Bytes
$History: * selfimage_mainform.ddp - 2005-11-18 1:21:36 PM - 51 Bytes
$History: * Add test for FAT32 filesystem when writing image files 
$History: * (can't have more than 4GB) and a test for enough free space in general
$History: **** V 0.13 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-18 2:27:30 PM - 35985 Bytes
$History: * selfimage_mainform.dfm - 2005-11-18 2:25:46 PM - 11812 Bytes
$History: * selfimage_mainform.h - 2005-11-18 2:26:32 PM - 5036 Bytes
$History: * selfimage_mainform.ddp - 2005-11-18 2:26:42 PM - 51 Bytes
$History: * Add help button
$History: **** V 0.14 by kfitzner ****
$History: * selfimage_mainform.cpp - 2005-11-18 3:54:59 PM - 36354 Bytes
$History: * selfimage_mainform.dfm - 2005-11-18 3:47:12 PM - 11819 Bytes
$History: * selfimage_mainform.h - 2005-11-18 2:26:32 PM - 5036 Bytes
$History: * selfimage_mainform.ddp - 2005-11-18 3:47:24 PM - 51 Bytes
$History: * Add BETA 2 tag to caption
$History: **** Latest ** V 0.15 by kfitzner ** 2005-11-20 8:01:28 PM ****
$History: * Remove BETA 2 from caption
*/
//----------------------------- $NoKeywords ---------------------------------

//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 2 Aug 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// The main read/write loop of this program uses Windows IO overlapping for
// asynchronous writes.  I have found that this doesn't significantly alter
// the throughput at all, but it does for some reason reduce the CPU
// overhead.
//---------------------------------------------------------------------------
#include <vcl.h>
#include <winnt.h>
#include <IdGlobal.hpp>
#pragma hdrstop

#include "SelfImage_MainForm.h"
#include "SelfImage_Exceptions.h"
#include "SelfImage_Utility.h"
#include "SelfImage_TImageStore.h"
#include "SelfImage_Preferences.h"
#include "SelfImage_AboutBox.h"
#include "TConfiguration.h"
#include "TProgramLog.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
//---------------------------------------------------------------------------
TMainForm *MainForm = NULL;
extern TConfiguration *SelfImageConfig;
extern TProgramLog    *Log;
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Main form constructor
//
__fastcall TMainForm::TMainForm(TComponent* Owner) : TForm(Owner)
{
  CurrentMode = modeSetup;
  DriveList = NULL;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnShow event handler - The form's objects are fully constructed but the
// form is not yet painted when this event occurs, so this is a great place
// to put code to initialize the initial state of the form's components.
//
void __fastcall TMainForm::FormShow(TObject *Sender)
{
  if (SelfImageConfig->Values["Remember Positions"]) {
    int nNewTop, nNewLeft, nNewWidth;
    nNewTop   = SelfImageConfig->Values["WindowTop"];
    nNewLeft  = SelfImageConfig->Values["WindowLeft"];
    nNewWidth = SelfImageConfig->Values["WindowWidth"];
    if (nNewTop != -1)
      Top = nNewTop;
    if (nNewLeft != -1)
      Left = nNewLeft;
    if (nNewWidth != -1)
      Width = nNewWidth;
  }  // if (SelfImageConfig->Values["Remember Positions"])
  RefreshDriveList();
  edtSourceFile->InitialDir = SelfImageConfig->Values["Personal"];
  edtTargetFile->InitialDir = SelfImageConfig->Values["Personal"];
  // cmbTargetDriveChange(NULL);
}  // void __fastcall TMainForm::FormShow(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnClose event handler - Save the position of the main form
//
void __fastcall TMainForm::FormClose(TObject *Sender, TCloseAction &Action)
{
  if (SelfImageConfig->Values["Remember Positions"]) {
    SelfImageConfig->Values["WindowTop"] = Top;
    SelfImageConfig->Values["WindowLeft"] = Left;
    SelfImageConfig->Values["WindowWidth"] = Width;
  }  // if (SelfImageConfig->Values["Remember Positions"])
}  // void __fastcall TMainForm::FormClose(TObject *Sender, TCloseAction &Action)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnClick handler for the input "File" radio button
//
void __fastcall TMainForm::radioSourceFileClick(TObject *Sender)
{
  __int64 nnSize;
  edtSourceFile->DoClick();
  if (FileExists(edtSourceFile->FileName) && (nnSize = FileSizeByName(edtSourceFile->FileName)) != 0) {
    this->SourceSize = nnSize;
    lblSourceSizeValue->Caption = MakeByteLabel(nnSize);
  } else {
    radioSourceFile->Checked = false;
    lblSourceSizeValue->Caption = "0";
  }  // else if (not valid file)
  SetVisualDefaults(NULL);
}  // void __fastcall TMainForm::radioSourceFileClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Onclick handler for the output "File" radio button
//
void __fastcall TMainForm::radioTargetFileClick(TObject *Sender)
{
  edtTargetFile->DoClick();
  if (edtTargetFile->FileName.IsEmpty())
    radioTargetFile->Checked = false;
  SetVisualDefaults(NULL);
}  // void __fastcall TMainForm::radioTargetFileClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnClick handler for the input "Drive" radio button and combo box
//
void __fastcall TMainForm::cmbSourceDriveChange(TObject *Sender)
{
  TDriveInfo *Drive = (TDriveInfo *)cmbSourceDrive->Items->Objects[cmbSourceDrive->ItemIndex];
  if (Drive->DriveType == driveFloppy)
    Drive->Refresh();
  if (Drive->Readable)
    lblSourceSizeValue->Caption = MakeByteLabel(Drive->Bytes);
  else
    lblSourceSizeValue->Caption = "No Disk";
  SourceSize = Drive->Bytes;
  SetVisualDefaults(NULL);
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnClick handler for the output "Drive" radio button
//
void __fastcall TMainForm::cmbTargetDriveChange(TObject *Sender)
{
  TDriveInfo *Drive = (TDriveInfo *)cmbTargetDrive->Items->Objects[cmbTargetDrive->ItemIndex];
  if (Drive->DriveType == driveFloppy)
    Drive->Refresh();
  if (Drive->Writable)
    lblTargetDriveSizeValue->Caption = MakeByteLabel(Drive->Bytes);
  else
    lblTargetDriveSizeValue->Caption = "No Disk";
  SetVisualDefaults(NULL);
}  // void __fastcall TMainForm::cmbTargetDriveChange(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// This method sets which visual controls are enabled/disabled based on
// user inputs
//
void __fastcall TMainForm::SetVisualDefaults(TObject *Sender)
{
  TDriveInfo *SourceDrive = NULL, *TargetDrive = NULL;

  if (radioSourceDrive->Checked && cmbSourceDrive->ItemIndex != -1)
    SourceDrive = (TDriveInfo *)cmbSourceDrive->Items->Objects[cmbSourceDrive->ItemIndex];
  if (radioTargetDrive->Checked && cmbTargetDrive->ItemIndex != -1)
    TargetDrive = (TDriveInfo *)cmbTargetDrive->Items->Objects[cmbTargetDrive->ItemIndex];

  edtSourceFile->Enabled = radioSourceFile->Checked;
  cmbSourceDrive->Enabled = radioSourceDrive->Checked;
  lblSourceSize->Enabled  = !lblSourceSizeValue->Caption.IsEmpty() && lblSourceSizeValue->Caption != "0";
  lblSourceSizeValue->Enabled = lblSourceSize->Enabled;
  edtTargetFile->Enabled = radioTargetFile->Checked;
  cmbTargetDrive->Enabled = radioTargetDrive->Checked;
  lblTargetDriveSize->Enabled = radioTargetDrive->Checked;
  lblTargetDriveSizeValue->Enabled = radioTargetDrive->Checked;
  btnOk->Enabled = (radioSourceFile->Checked && !edtSourceFile->FileName.IsEmpty() || radioSourceDrive->Checked && SourceDrive)
                    && (radioTargetFile->Checked && !edtTargetFile->FileName.IsEmpty() || radioTargetDrive->Checked && TargetDrive)
                    && !(radioSourceDrive->Checked && radioTargetDrive->Checked && cmbSourceDrive->Text == cmbTargetDrive->Text)
                    && !(radioSourceDrive->Checked && SourceDrive && !SourceDrive->Readable)
                    && !(radioTargetDrive->Checked && TargetDrive && !TargetDrive->Writable);
  menu_File_Start->Enabled = btnOk->Enabled;
}  // void __fastcall TMainForm::SetVisualDefaults(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Ok button OnClick event handler - the user clicked Ok, so get to work
//
void __fastcall TMainForm::btnOkClick(TObject *Sender)
{
  int ChunkSize;

  // Perform last-minute sanity checks and give the user a chance to back out if something doesn't make sense
  if (!SanityCheck())
    return;

  // Create the input image store
  if (radioSourceFile->Checked) {
    SourceImage = new TImageStore(isFile, edtSourceFile->FileName);
  } else {  // if source drive
    TDriveInfo *SourceDrive = (TDriveInfo *)cmbSourceDrive->Items->Objects[cmbSourceDrive->ItemIndex];
    SourceImage = new TImageStore(isDrive, SourceDrive->DeviceName);
  }  // else if source drive

  // Create the output image store
  if (radioTargetFile->Checked) {
    TargetImage = new TImageStore(isFile, edtTargetFile->FileName, true);
  } else {  // if source drive
    TDriveInfo *TargetDrive = (TDriveInfo *)cmbTargetDrive->Items->Objects[cmbTargetDrive->ItemIndex];
    TargetImage = new TImageStore(isDrive, TargetDrive->DeviceName, true);
  }  // else if source drive

  if (SourceImage->Geometry.Bytes < 16 * 1024 * 1024)
    ChunkSize = 64 * 1024;
  else
    ChunkSize = 1024 * 1024;

  Buffer = new TImageBuffer(ChunkSize, 16);

  ReadThread = new TSelfImageReadThread(SourceImage, Buffer);
  WriteThread = new TSelfImageWriteThread(TargetImage, Buffer);

  nnTotalBytes = SourceImage->Geometry.Bytes;
  VisualMode(modeOperating);
  UpdateStatus(true);

  ReadThread->Resume();
  WriteThread->Resume();

  UpdateTimer->Enabled = true;

/*
  // This is the old way of copying.
  HANDLE hInfile  = INVALID_HANDLE_VALUE;
  HANDLE hOutfile = INVALID_HANDLE_VALUE;
  OVERLAPPED Overlap;
  AnsiString sInFileDevice;
  void *buffer[2] = {NULL,NULL};
  unsigned nBufferSize;
  int nCurrent = 0;  // Current buffer
  int nPrevious;     // Previous buffer
  DWORD nBytesRead[2], nBytesWritten;
  DWORD nProcessMessagesFlag = 8;
  bool bWriteOk, bReadOk;
  TDrive *Drive;

  try {
    bCancelOperation = false;
    BytesWrittenTotal.QuadPart = 0;
    memset(&Overlap, 0, sizeof(Overlap));

    Drive = (TDrive *)cmbDriveList->Items->Objects[cmbDriveList->ItemIndex];
    if (Drive->DriveType == driveFixed)
      nBufferSize = nBytesPerSector * 8 * 64;
    else
      nBufferSize = nBytesPerSector * 36;
      
    if (Drive->MountPoint.IsEmpty())
      hInfile = NTOpen(Drive->DeviceName, GENERIC_READ | SYNCHRONIZE, FILE_ATTRIBUTE_NORMAL, FILE_SHARE_READ | FILE_SHARE_WRITE, FILE_OPEN, FILE_SYNCHRONOUS_IO_NONALERT | FILE_SEQUENTIAL_ONLY);
    else
      hInfile = CreateFile(Drive->DeviceName.c_str(), GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN | FILE_FLAG_BACKUP_SEMANTICS, NULL);
    if (hInfile == INVALID_HANDLE_VALUE)
      throw ESelfImageFileError("SelfImage could not open drive " + Drive->DisplayName + ": '" + GetLastErrorMessage() +"'");

    buffer[0] = VirtualAlloc(NULL, nBufferSize, MEM_COMMIT, PAGE_READWRITE);
    if (!buffer[0])
      throw ESelfImageAllocError("SelfImage could not allocate memory for its internal disk read buffer: '" + GetLastErrorMessage() + "'");
    buffer[1] = VirtualAlloc(NULL, nBufferSize, MEM_COMMIT, PAGE_READWRITE);
    if (!buffer[1])
      throw ESelfImageAllocError("SelfImage could not allocate memory for its internal disk read buffer: '" + GetLastErrorMessage() + "'");

    hOutfile = CreateFile(edtTargetFile->FileName.c_str(), GENERIC_WRITE, 0, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL);
    if (hOutfile == INVALID_HANDLE_VALUE)
      throw ESelfImageFileError("SelfImage could not open the file " + edtTargetFile->FileName + " for output: '" + GetLastErrorMessage() + "'");

    SetFilePointer(hInfile, 0, NULL, FILE_BEGIN);
    SetFilePointer(hOutfile, 0, NULL, FILE_BEGIN);
    SetEndOfFile(hOutfile);

    VisualMode(modeOperating);
    UpdateStatus(true);

    if (!ReadFile(hInfile, buffer[nCurrent], nBufferSize, &nBytesRead[nCurrent], NULL))
      throw ESelfImageFileError("SelfImage encountered a problem reading from drive " + Drive->DisplayName + ": " + GetLastErrorMessage() );
    WriteFile(hOutfile, buffer[nCurrent], nBytesRead[nCurrent], NULL, &Overlap);
    BytesReadTotal.QuadPart = nBytesRead[nCurrent];
    nPrevious = nCurrent; nCurrent ^= 1;
    ReadLoop:
    while ( (bReadOk = ReadFile(hInfile, buffer[nCurrent], nBufferSize, &nBytesRead[nCurrent], NULL)) == true && nBytesRead[nCurrent] > 0 && !bCancelOperation) {
      BytesReadTotal.QuadPart += nBytesRead[nCurrent];
      bWriteOk = GetOverlappedResult(hOutfile, &Overlap, &nBytesWritten, TRUE);
      BytesWrittenTotal.QuadPart += nBytesWritten;
      Overlap.Offset = BytesWrittenTotal.LowPart;
      Overlap.OffsetHigh = BytesWrittenTotal.HighPart;
      if (!bWriteOk || nBytesWritten != nBytesRead[nPrevious])
        throw ESelfImageFileError("SelfImage encountered a problem writing to the file " + edtTargetFile->FileName + ".  The operation wrote " + nBytesWritten + "/" + nBytesRead[nPrevious] + "bytes (" + BytesWrittenTotal.QuadPart + "/" + nnTotalBytes + " total bytes written): '" + GetLastErrorMessage() + "'.");
      WriteFile(hOutfile, buffer[nCurrent], nBytesRead[nCurrent], NULL, &Overlap);
      nPrevious = nCurrent; nCurrent ^= 1;
      if (--nProcessMessagesFlag == 0) {
        nProcessMessagesFlag = 8;
        UpdateStatus();
        Application->ProcessMessages();
      }  // if (ProcessMessagesFlag == 0)
    }  // while (ReadFile(hInfile, buffer, nBufferSize, &nBytesRead, NULL) && nBytesRead > 0)
    // Devices opened with NtCreateFile() don't handle partial reads properly.  When a read fails, we have to step
    // down the read size to the sector size and try again.  This is a hack - there is probably a better way to do
    // this that involves understanding NtCreateFile() better.
    if (!bCancelOperation && nBufferSize > nBytesPerSector) {
      nBufferSize = nBytesPerSector;
      goto ReadLoop;
    }  // if (!bReadOk && nBufferSize > nBytesPerSector)
    if (!bCancelOperation) {
      bWriteOk = GetOverlappedResult(hOutfile, &Overlap, &nBytesWritten, TRUE);
      BytesWrittenTotal.QuadPart += nBytesWritten;
      if (!bWriteOk || nBytesWritten != nBytesRead[nPrevious])
        throw ESelfImageFileError("SelfImage encountered a problem writing to the file " + edtTargetFile->FileName + ".  The operation wrote " + nBytesWritten + "/" + nBytesRead[nPrevious] + "bytes (" + BytesWrittenTotal.QuadPart + "/" + nnTotalBytes + " total bytes written): '" + GetLastErrorMessage() + "'.");
    }  // if (!bCancelOperation)
//  if (!bReadOk)  // This isn't working properly for devices opened with NtCreateFile()
//    throw ESelfImageFileError("SelfImage encountered a problem reading from drive " + Drive->DisplayName + ": " + GetLastErrorMessage() );
    if (!bCancelOperation && BytesWrittenTotal.QuadPart != BytesReadTotal.QuadPart)
      throw ESelfImageFileError("SelfImage's internal counters of how much it read and how much it wrote don't match.");
    if (!bCancelOperation && BytesWrittenTotal.QuadPart < nnTotalBytes && !bReadOk)
      throw ESelfImageFileError("SelfImage encountered a problem reading from the device: " + GetLastErrorMessage());
    UpdateStatus(true);
  }  // try

  __finally {
    VisualMode(modeSetup);
    if (hOutfile != INVALID_HANDLE_VALUE)
      CloseHandle(hOutfile);
    if (buffer)
      VirtualFree(buffer, 0, MEM_RELEASE);
    if (hInfile != INVALID_HANDLE_VALUE)
      CloseHandle(hInfile);
  }  // __finally
*/
}  // void __fastcall TMainForm::btnOkClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnCancel OnClick event - the user pressed the cancel button.
//
void __fastcall TMainForm::btnCancelClick(TObject *Sender)
{
  if (CurrentMode == modeSetup) {
    Close();
  } else {
    ReadThread->Terminate();
    WriteThread->Terminate();
  }  // else (VisualMode != modeSetup)
}  // void __fastcall TMainForm::btnCancelClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// File->Exit menu item OnClick event handler
//
void __fastcall TMainForm::menu_File_ExitClick(TObject *Sender)
{
  Close();
}  // void __fastcall TMainForm::menu_File_ExitClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Edit->Refresh Drive Lists menu item OnClick event handler
//
void __fastcall TMainForm::menu_Edit_RefreshClick(TObject *Sender)
{
  RefreshDriveList();
}  // void __fastcall TMainForm::menu_Edit_RefreshClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Edit->Preferences menu item OnClick event handler
//
void __fastcall TMainForm::menu_Edit_PreferencesClick(TObject *Sender)
{
  formSelfImagePreferences = new TformSelfImagePreferences(Application);
  formSelfImagePreferences->ShowModal();
  delete formSelfImagePreferences;
}  // void __fastcall TMainForm::menu_Edit_PreferencesClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Help->* menu items OnClick event handler
// Launch the help application - help page is stored in the menu item's
// HelpContext field
//
void __fastcall TMainForm::menu_Help_ItemsClick(TObject *Sender)
{
  Application->HelpContext(((TMenuItem *)Sender)->HelpContext);
}  // void __fastcall TMainForm::menu_Help_OverviewClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Help->About menu item OnClick event handler
// Show the about box
//
void __fastcall TMainForm::menu_Help_AboutClick(TObject *Sender)
{
  formSelfImageAboutBox = new TformSelfImageAboutBox(Application);
  formSelfImageAboutBox->ShowModal();
  delete formSelfImageAboutBox;
}  // void __fastcall TMainForm::About1Click(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnHelp OnClick event handler
//
void __fastcall TMainForm::btnHelpClick(TObject *Sender)
{
  Application->HelpContext(((TControl *)Sender)->HelpContext);
}  // void __fastcall TMainForm::btnHelpClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// UpdateTimer OnTimer event handler - the UpdateTimer timer is what
// determines when visual updates occur when a copy is in progress.  It
// also checks to see if any errors have occured during the copy and cleans
// up if one does.
//
void __fastcall TMainForm::UpdateTimerTimer(TObject *Sender)
{
  if (WriteThread->Finished || ReadThread->Finished && ReadThread->ErrorFlag) {
    UpdateTimer->Enabled = false;
    if (!WriteThread->Finished) {
      WriteThread->Terminate();
      WriteThread->WaitFor();
    }  // if (!WriteThread->Finished)
    if (!ReadThread->Finished) {
      ReadThread->Terminate();
      ReadThread->WaitFor();
    }  // if (!WriteThread->CleanExit)
    if (ReadThread->ErrorFlag || WriteThread->ErrorFlag) {
      AnsiString ErrorMessage;
      if (ReadThread->ErrorFlag)
        ErrorMessage = ReadThread->ErrorMessage + (WriteThread->ErrorFlag?"\r\n":"");
      if (WriteThread->ErrorFlag)
        ErrorMessage += WriteThread->ErrorMessage;
      Application->MessageBox(ErrorMessage.c_str(), "Image copy error", MB_OK | MB_ICONERROR);
    }  // if (ReadThread->ErrorFlag || WriteThread->ErrorFlag)
    delete SourceImage;
    delete TargetImage;
    delete Buffer;
    delete ReadThread;
    delete WriteThread;
    VisualMode(modeSetup);
  }  // if (ReadThread->CleanExit || WriteThread->CleanExit)
  else
    UpdateStatus();
}  // void __fastcall TMainForm::UpdateTimerTimer(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Read (or re-read) the information for all the drive partitions, including
// which ones are writable, and set up the drive combo boxes.
//
void __fastcall TMainForm::RefreshDriveList(void)
{
  cmbSourceDrive->Clear();
  cmbTargetDrive->Clear();
  if (DriveList)
    delete DriveList;
  radioSourceDrive->Checked = false;
  radioTargetDrive->Checked = false;

  DriveList = new TDriveList(true);
  for (int n = 0; n < DriveList->Count; n++) {
    TDriveInfo *Drive = DriveList->Items[n];
    // cmbDriveList->ItemsEx->AddItem(Drive->DisplayName, 0, 0, 0, 0, Drive);
    cmbSourceDrive->AddItem(Drive->DisplayName, (TObject *)Drive);
    if (Drive->Writable)
      cmbTargetDrive->AddItem(Drive->DisplayName, (TObject *)Drive);
  }  // for (int n = 0; n < DriveList->Count; n++)
  // If there are no writable partitions then disable the output drive radio button
  if (!cmbTargetDrive->Items->Count)
    radioTargetDrive->Enabled = false;
  for (int n = 0; n < cmbSourceDrive->Items->Count; n++) {
    TDriveInfo *Drive = (TDriveInfo *)cmbSourceDrive->Items->Objects[n];
    if (Drive->MountPoint.Pos("C:")) {
      cmbSourceDrive->ItemIndex = n;
      break;
    }  // if (Drive->MountPoint.Pos("C:"))
  }  // for (int n = 0; n < cmbDriveList->Items->Count; n++)
}  // void __fastcall RefreshDriveList(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Switch visual modes between setup (user is setting up the copy) and
// operating (the copy is occuring).
//
void __fastcall TMainForm::VisualMode(TMainFormVisualMode Mode)
{
  if (Mode == modeSetup) {
    btnOk->Enabled = !edtTargetFile->FileName.IsEmpty();
    menu_File->Enabled = true;
    menu_Edit->Enabled = true;
    menu_Help->Enabled = true;
    gbStatus->Visible = false;
    gbStatus->SendToBack();
  }  // if (Mode == modeSetup)

  if (Mode == modeOperating) {
    btnOk->Enabled = false;
    menu_File->Enabled = false;
    menu_Edit->Enabled = false;
    menu_Help->Enabled = false;
    gbStatus->Visible = true;
    gbStatus->BringToFront();
    ProgressBar->Maximum = ProgressBar->Width;
    ProgressBar->Position = 0;
    lblBytesTotalValue->Caption = MakeByteLabel(nnTotalBytes);
    lblBytesWrittenValue->Caption = "0";
  }  // if (Mode == modeOperating)
  CurrentMode = Mode;
  return;
}  // void __fastcall TMainForm::VisualMode(TMainFormVisualMode Mode)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TMainForm::UpdateStatus(bool bInit)
{
  static LARGE_INTEGER StartTimer, LastTimer, PerformanceFrequency;
  static __int64 nnLastBytesWrittenTotal;
  __int64 nnBytesWrittenTotal;
  LARGE_INTEGER Timer;
  long double ddTimeDifference, ddBytesDifference;
  int nProgressTicks;

  if (bInit) {
    QueryPerformanceFrequency(&PerformanceFrequency);
    QueryPerformanceCounter(&StartTimer);
    LastTimer.QuadPart = StartTimer.QuadPart;
    nnLastBytesWrittenTotal = 0;
  } else {  // if (bInit)
    nnBytesWrittenTotal = WriteThread->BytesWritten;
    nProgressTicks = nnBytesWrittenTotal * ProgressBar->Width / nnTotalBytes;
    ProgressBar->Position = nProgressTicks;
    lblBytesWrittenValue->Caption = MakeByteLabel(nnBytesWrittenTotal);
    QueryPerformanceCounter(&Timer);
    ddTimeDifference = Timer.QuadPart - LastTimer.QuadPart;
    ddTimeDifference /= PerformanceFrequency.QuadPart;
    if (ddTimeDifference > 1.0) {
      ddBytesDifference = nnBytesWrittenTotal - nnLastBytesWrittenTotal;
      lblWriteSpeedValue->Caption = MakeByteLabel(ddBytesDifference/ddTimeDifference) + "/s";
      ddBytesDifference = nnBytesWrittenTotal;
      ddTimeDifference = Timer.QuadPart - StartTimer.QuadPart;
      ddTimeDifference /= PerformanceFrequency.QuadPart;
      lblOverallSpeedValue->Caption = MakeByteLabel(ddBytesDifference/ddTimeDifference) + "/s";
      LastTimer.QuadPart = Timer.QuadPart;
      nnLastBytesWrittenTotal = nnBytesWrittenTotal;
    }  // if (dTimeDifference > 1.0)
  }  // if (bInit)
}  // void __fastcall StatusUpdateTimer(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Perform a sanity check on what the user is doing.  This is the last line
// of defense to prevent the user from doing something stupid.  Presents
// a warning, but will let the user continue.
//
bool __fastcall TMainForm::SanityCheck(void)
{
  TDriveInfo /* *SourceDrive = NULL, */ *TargetDrive = NULL;
  AnsiString Warning;
  AnsiString FileSystem;

//if (radioSourceDrive->Checked && cmbSourceDrive->ItemIndex != -1)
//  SourceDrive = (TDriveInfo *)cmbSourceDrive->Items->Objects[cmbSourceDrive->ItemIndex];
  if (radioTargetDrive->Checked && cmbTargetDrive->ItemIndex != -1)
    TargetDrive = (TDriveInfo *)cmbTargetDrive->Items->Objects[cmbTargetDrive->ItemIndex];

  // Input is a file, Output is a partition, and input size is larger than the target
  if (radioSourceFile->Checked && radioTargetDrive->Checked && TargetDrive && SourceSize > TargetDrive->Bytes)
    Warning = "The input image is " + MakeByteLabel(SourceSize - TargetDrive->Bytes) + " larger than the partition it is being written to.  SelfImage will not write past the end of the target disk partition, but it is likely that this image will not work properly in the target partition.  Do not proceed unless you know what you are doing is correct.";
  // Input is a file, Output is a partition and input size is smaller than the target
  else if (radioSourceFile->Checked && radioTargetDrive->Checked && TargetDrive && SourceSize < TargetDrive->Bytes)
    Warning = "The input image is " + MakeByteLabel(TargetDrive->Bytes - SourceSize) + " smaller than the partition it is being written to.  This image did not come from the partition you are writing it to - writing it to this partition may not have the effect you wish.  Do not proceed unless you know what you are doing is correct.";
  // Input is anything, Output is a file, not enough space on destination device
  else if (radioTargetFile->Checked && SourceSize > GetFreeDiskSpace(edtTargetFile->FileName))
    Warning = "Not enough space on destination volume.  Input size is " + MakeByteLabel(SourceSize) + ", available space is " + MakeByteLabel(GetFreeDiskSpace(edtTargetFile->FileName)) + ".";
  // Input is > 4GB, Output is file on a FAT32 partition
  else if (radioTargetFile->Checked && SourceSize > 4294967296ui64 && GetFileSystemType(edtTargetFile->FileName).SubString(1,3) == "FAT")
    Warning = "The source for this image is more than 4GB and the target is on a FAT/FAT32 volume that cannot store files larger than 4GB.  If you continue, the image will be limited to 4GB and will be dangerous to use.";
  // Copying one partition to another
  else if (radioSourceDrive->Checked && radioTargetDrive->Checked)
    // Source partition is larger than the target - really bad news
    if (SourceSize > TargetDrive->Bytes)
      Warning = "You are copying directly from one partition to another and the source partition is larger than the target.  SelfImage will not write past the end of the target disk partition, but it is likely that this operation will not work as intended.  Even if the the disk partition is mountable in an operating system, it is likely that data corruption will occur.  Do not proceed unless you know what you are doing is correct.";
    // Source is smaller than the target - sometimes bad news
    else if (SourceSize < TargetDrive->Bytes)
      Warning = "You are copying directly from one partition to another and the source partition is smaller than the target.  Do not proceed unless you know what you are doing is correct.";
    // Partitions are identical size, but the user may have made a mistake - most users never need to do this
    else
      Warning = "You are copying directly from one partition to another.  Make sure this is what you intended.";

  FileSystem = GetFileSystemType(edtTargetFile->FileName);

  // Warn when writing directly to a disk partition
  if (!SelfImageConfig->Values["NoDirectWriteWarn"])
    if (radioTargetDrive->Checked) {
      if (!Warning.IsEmpty())
        Warning += "\r\n\r\n";
      Warning += "You are writing directly to a disk partition.  ALL EXISTING DATA ON THE PARTITION WILL BE OVERWRITTEN!";
    }  // if (radioSourceDrive->Checked && radioTargetDrive->Checked)

  if (!Warning.IsEmpty()) {
    Warning += "\r\n\r\nAre you SURE you want to continue?";
    if (Application->MessageBox(Warning.c_str(), "WARNING  WARNING  WARNING", MB_YESNO | MB_ICONWARNING) == IDNO)
      return false;
  }  // if (!Warning.IsEmpty())

  return true;
}  // bool __fastcall TMainForm::SanityCheck(void)
//---------------------------------------------------------------------------

