/* rungpg.c 
 *	Copyright (C) 2000, 2001 Werner Koch (dd9jn), g10 Code GmbH
 *	Copyright (C) 2002, 2003, 2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <signal.h>
#include <fcntl.h>
#include <windows.h>

#include "gpgme.h"
#include "util.h"
#include "ops.h"
#include "wait.h"
#include "rungpg.h"
#include "context.h"  /*temp hack until we have gpgme_dat_t methods to do I/O */
#include "gpgme-io.h"
#include "sema.h"

#include "status-table.h"

/* This type is used to build a list of gpg arguments and
 * data sources/sinks */
struct arg_and_data_s {
    struct arg_and_data_s * next;
    gpgme_data_t data;/* If this is not NULL .. */
    int dup_to;
    int print_fd;     /* print the fd number and not the special form of it */
    char arg[1];      /* .. this is used */
};

struct fd_data_map_s {
    gpgme_data_t data;
    int inbound;  /* true if this is used for reading from gpg */
    int dup_to;
    int fd;       /* the fd to use */
    int peer_fd;  /* the outher side of the pipe */
};


struct gpg_object_s {
    struct arg_and_data_s * arglist;
    struct arg_and_data_s ** argtail;
    int arg_error;  

    struct {
        int fd[2];
	int eof;
        size_t bufsize;
        char * buffer;
        size_t readpos;
        gpg_status_handler_t fnc;
        void * fnc_value;
    } status;

    struct {
	int fd[2];
	int eof;
	int enabled;
	size_t bufsize;
	char * buffer;
	void * fnc_value;
    } logging;

    /* This is a kludge - see the comment at gpg_colon_line_handler */
    struct {
        int fd[2]; 
	int eof;
        size_t bufsize;
        char *buffer;
        size_t readpos;
        gpg_colon_line_handler_t fnc;  /* this indicate use of this structrue */
        void *fnc_value;
        int simple;
    } colon;

    char **argv;  
    struct fd_data_map_s *fd_data_map;

    int pid; /* we can't use pid_t because we don't use it in Windows */

    int running;

    char *path;

    /* stuff needed for interactive (command) mode */
    struct {
        int used;
        int fd;
        gpgme_data_t cb_data;   /* hack to get init the above fd later */
        gpg_status_code_t code;  /* last code */
        char *keyword;       /* what has been requested (malloced) */
        gpg_command_handler_t fnc; 
        void *fnc_value;
    } cmd;
};

struct reap_s {
    struct reap_s *next;
    int pid;
    time_t entered;
    int term_send;
};

static struct reap_s *reap_list;
DEFINE_STATIC_LOCK (reap_list_lock);


static void free_argv ( char **argv );
static void free_fd_data_map ( struct fd_data_map_s *fd_data_map );

static int gpg_inbound_handler ( void *opaque, int pid, int fd );
static int gpg_outbound_handler ( void *opaque, int pid, int fd );

static int gpg_logging_handler( void * opaque, int pid, int fd );
static gpgme_error_t read_logging( _gpg_object_t gpg );

static int gpg_status_handler( void *opaque, int pid, int fd );
static gpgme_error_t read_status( _gpg_object_t gpg );

static int gpg_colon_line_handler ( void *opaque, int pid, int fd );
static gpgme_error_t read_colon_line ( _gpg_object_t gpg );

static int command_cb ( void *opaque,
                        char *buffer, size_t length, size_t *nread );


void 
rungpg_cleanup (void)
{
    DESTROY_LOCK (reap_list_lock);
}


static void
close_notify_handler ( int fd, void *opaque )
{
    _gpg_object_t gpg = opaque;

    assert( fd != -1 );
    if( gpg->status.fd[0] == fd )
        gpg->status.fd[0] = -1;
    else if (gpg->status.fd[1] == fd )
        gpg->status.fd[1] = -1;
    else if( gpg->logging.fd[0] == fd )
	gpg->logging.fd[0] = -1;
    else if( gpg->logging.fd[1] == fd )
	gpg->logging.fd[1] = -1;
    else if (gpg->colon.fd[0] == fd )
        gpg->colon.fd[0] = -1;
    else if (gpg->colon.fd[1] == fd )
        gpg->colon.fd[1] = -1;
    else if (gpg->fd_data_map) {
        int i;

        for (i=0; gpg->fd_data_map[i].data; i++ ) {
            if ( gpg->fd_data_map[i].fd == fd ) {
                gpg->fd_data_map[i].fd = -1;
                break;
            }
            if ( gpg->fd_data_map[i].peer_fd == fd ) {
                gpg->fd_data_map[i].peer_fd = -1;
                break;
            }
        }
    }
}


gpgme_error_t
_gpgme_gpg_new (_gpg_object_t *r_gpg)
{
    _gpg_object_t gpg;
    char * p;
    char buf[25];
    int rc = 0;

    gpg = calloc ( 1, sizeof *gpg );
    if ( !gpg ) {
        rc = mk_error (Out_Of_Core);
        goto leave;
    }
    gpg->argtail = &gpg->arglist;

    gpg->status.fd[0] = -1;
    gpg->status.fd[1] = -1;
    gpg->logging.fd[0] = -1;
    gpg->logging.fd[1] = -1;
    gpg->colon.fd[0] = -1;
    gpg->colon.fd[1] = -1;
    gpg->cmd.fd = -1;

    gpg->pid = -1;

    /* allocate the read buffer for the status pipe */
    gpg->status.bufsize = 1024;
    gpg->status.readpos = 0;
    p = gpg->status.buffer = malloc( gpg->status.bufsize );
    if( !p ) {
        rc = mk_error (Out_Of_Core);
        goto leave;
    }

    gpg->logging.bufsize = 512;
    p = gpg->logging.buffer = malloc( gpg->logging.bufsize+4 );
    if( !p ) {
	rc = mk_error( Out_Of_Core );
	goto leave;
    }

    /* In any case we need a status pipe - create it right here  and
     * don't handle it with our generic gpgme_data_t mechanism */
    if( _gpgme_io_pipe( gpg->status.fd, 1 ) == -1
     || _gpgme_io_pipe( gpg->logging.fd, 1 ) == -1 ) {
        rc = mk_error (Pipe_Error);
        goto leave;
    }
    if ( _gpgme_io_set_close_notify( gpg->status.fd[0],
                                     close_notify_handler, gpg )
         || _gpgme_io_set_close_notify( gpg->status.fd[1],
                                        close_notify_handler, gpg )
	 || _gpgme_io_set_close_notify( gpg->logging.fd[0],
					 close_notify_handler, gpg )
	 || _gpgme_io_set_close_notify( gpg->logging.fd[1],
					close_notify_handler, gpg ) ) {
        rc = mk_error (General_Error);
        goto leave;
    }

    gpg->status.eof = 0;
    _gpgme_gpg_add_arg( gpg, "--status-fd" );
    sprintf( buf, "%d", gpg->status.fd[1] );
    _gpgme_gpg_add_arg ( gpg, buf );

    gpg->logging.eof = 0;
    _gpgme_gpg_add_arg( gpg, "--logger-fd" );
    sprintf( buf, "%d", gpg->logging.fd[1] );
    _gpgme_gpg_add_arg( gpg, buf );

    _gpgme_gpg_add_arg ( gpg, "--no-tty" );


 leave:
    if( rc ) {
        _gpgme_gpg_release( &gpg );
        *r_gpg = NULL;
    }
    else
        *r_gpg = gpg;
    return rc;
}

static void
_pipe_close (int * fd)
{
    if (fd[0] != -1)
	_gpgme_io_close (fd[0]);
    if (fd[1] != -1)
	_gpgme_io_close (fd[1]);
}


void
_gpgme_gpg_release (_gpg_object_t *r_gpg)
{
    struct arg_and_data_s * a;
    _gpg_object_t gpg = *r_gpg;

    if( !gpg )
        return;
    safe_free (gpg->status.buffer);
    safe_free (gpg->logging.buffer);
    safe_free (gpg->colon.buffer); 
    safe_free (gpg->cmd.keyword);
    if (gpg->argv)
        free_argv(gpg->argv);

    safe_free (gpg->path);

    if (gpg->pid != -1)
        _gpgme_remove_proc_from_wait_queue ( gpg->pid );
    _pipe_close( gpg->status.fd );
    _pipe_close( gpg->colon.fd );
    _pipe_close( gpg->logging.fd );
    free_fd_data_map (gpg->fd_data_map);
    while (gpg->arglist) {
	a = gpg->arglist->next;
	safe_free (gpg->arglist);
	gpg->arglist = a;
    }
    if( gpg->running ) {
        int pid = gpg->pid;
        struct reap_s *r;

        /* resuse the memory, so that we don't need to allocate another
         * mem block and have to handle errors */
        assert( sizeof *r < sizeof *gpg );
        r = (void*)gpg;
        memset( r, 0, sizeof *r );
        r->pid = pid;
        r->entered = time( NULL );
        LOCK(reap_list_lock);
        r->next = reap_list;
        reap_list = r;
        UNLOCK(reap_list_lock);
    }
    else
        safe_free ( gpg );
    *r_gpg = NULL;
}


static void
do_reaping (void)
{
    struct reap_s *r, *rlast;
    static time_t last_check;
    time_t cur_time = time (NULL);

    /* a race does not matter here */
    if (!last_check)
        last_check = time(NULL);

    if (last_check >= cur_time)
        return;  /* we check only every second */

    /* fixme: it would be nice if to have a TRYLOCK here */
    LOCK (reap_list_lock);  
    for (r=reap_list,rlast=NULL; r ; rlast=r, r=r?r->next:NULL) {
        int dummy1, dummy2;

        if ( _gpgme_io_waitpid (r->pid, 0, &dummy1, &dummy2) ) {
            /* process has terminated - remove it from the queue */
            void *p = r;
            if (!rlast) {
                reap_list = r->next;
                r = reap_list;
            }
            else {
                rlast->next = r->next;
                r = rlast;
            }
            safe_free (p);
        }
        else if ( !r->term_send ) {
            if( r->entered+1 >= cur_time ) {
                _gpgme_io_kill ( r->pid, 0);
                r->term_send = 1;
                r->entered = cur_time;
            }
        }
        else {
            /* give it 5 second before we are going to send the killer */
            if ( r->entered+5 >= cur_time ) {
                _gpgme_io_kill (r->pid, 1);
                r->entered = cur_time; /* just in case we have to repat it */
            }
        }
    }
    UNLOCK (reap_list_lock);  
}

void
_gpgme_gpg_housecleaning ()
{
    do_reaping ();
}

    

gpgme_error_t
_gpgme_gpg_add_arg_concat (_gpg_object_t gpg, const char *arg1, const char *arg2)
{
    char * str;

    str = malloc (strlen (arg1) + strlen (arg2) + 2);	
    if (!str)
	return GPGME_Out_Of_Core;
    sprintf (str, "%s%s", arg1, arg2);
    _gpgme_gpg_add_arg (gpg, str);
    free (str);
    return 0;
}


gpgme_error_t
_gpgme_gpg_add_arg (_gpg_object_t gpg, const char *arg)
{
    struct arg_and_data_s *a;

    assert (gpg);
    assert (arg);

    a = malloc (sizeof *a + strlen (arg));
    if (!a) 
    {
        gpg->arg_error = 1;
        return mk_error (Out_Of_Core);
    }
    a->next = NULL;
    a->data = NULL;
    a->dup_to = -1;
    strcpy (a->arg, arg);
    *gpg->argtail = a;
    gpg->argtail = &a->next;
    return 0;
}

gpgme_error_t
_gpgme_gpg_add_data ( _gpg_object_t gpg, gpgme_data_t data, int dup_to )
{
    struct arg_and_data_s *a;

    assert (gpg);
    assert (data);

    a = malloc ( sizeof *a - 1 );
    if ( !a ) {
        gpg->arg_error = 1;
        return mk_error(Out_Of_Core);
    }
    a->next = NULL;
    a->data = data;
    if ( dup_to == -2 ) {
        a->print_fd = 1;
        a->dup_to = -1;
    }
    else {
        a->print_fd = 0;
        a->dup_to = dup_to;
    }
    *gpg->argtail = a;
    gpg->argtail = &a->next;
    return 0;
}


/*
 * Note, that the status_handler is allowed to modifiy the args value
 */
void
_gpgme_gpg_set_status_handler ( _gpg_object_t gpg,
                                gpg_status_handler_t fnc, void *fnc_value ) 
{
    assert (gpg);
    gpg->status.fnc = fnc;
    gpg->status.fnc_value = fnc_value;
}

/* Kludge to process --with-colon output */
gpgme_error_t
_gpgme_gpg_set_colon_line_handler ( _gpg_object_t gpg,
                                    gpg_colon_line_handler_t fnc, void *fnc_value ) 
{
    assert (gpg);

    gpg->colon.bufsize = 1024;
    gpg->colon.readpos = 0;
    gpg->colon.buffer = malloc (gpg->colon.bufsize);
    if (!gpg->colon.buffer) {
        return mk_error (Out_Of_Core);
    }
    if (_gpgme_io_pipe (gpg->colon.fd, 1) == -1) {
        safe_free (gpg->colon.buffer); gpg->colon.buffer = NULL;
        return mk_error (Pipe_Error);
    }
    if ( _gpgme_io_set_close_notify (gpg->colon.fd[0],
                                     close_notify_handler, gpg)
         ||  _gpgme_io_set_close_notify (gpg->colon.fd[1],
                                         close_notify_handler, gpg) ) {
        return mk_error (General_Error);
    }
    gpg->colon.eof = 0;
    gpg->colon.fnc = fnc;
    gpg->colon.fnc_value = fnc_value;
    gpg->colon.simple = 0;
    return 0;
}


gpgme_error_t
_gpgme_gpg_set_simple_line_handler ( _gpg_object_t gpg,
                                     gpg_colon_line_handler_t fnc,
                                     void *fnc_value ) 
{
    gpgme_error_t err;

    err = _gpgme_gpg_set_colon_line_handler (gpg, fnc, fnc_value);
    if (!err)
        gpg->colon.simple = 1;
    return err;
}


void
_gpgme_gpg_set_path (_gpg_object_t gpg, const char *homedir)
{
    const char * s = "gpg.exe";
    safe_free (gpg->path);
    if (!homedir)
	return;
    gpg->path = malloc (strlen (homedir) + 1 + strlen (s));
    sprintf (gpg->path, "%s\\%s", homedir, s);
}


gpgme_error_t
_gpgme_gpg_set_list_options (_gpg_object_t gpg, int opts)
{
    assert (gpg);
    if (opts) {
	_gpgme_gpg_add_arg (gpg, "--list-options");
	if (opts & GPGME_LISTOPT_SIGSUBPKT)
	    _gpgme_gpg_add_arg (gpg, "show-sig-subpackets");
    }
    return 0;
}

/* 
 * The Fnc will be called to get a value for one of the commands with
 * a key KEY.  If the Code pssed to FNC is 0, the function may release
 * resources associated with the returned value from another call.  To
 * match such a second call to a first call, the returned value from
 * the first call is passed as keyword.
 */

gpgme_error_t
_gpgme_gpg_set_command_handler( _gpg_object_t gpg,
                                gpg_command_handler_t fnc, void *fnc_value ) 
{
    gpgme_data_t tmp;
    gpgme_error_t err;

    assert( gpg );

    err = gpgme_data_new_with_read_cb( &tmp, command_cb, gpg );
    if( err )
        return err;
        
    _gpgme_gpg_add_arg ( gpg, "--command-fd" );
    _gpgme_gpg_add_data( gpg, tmp, -2 );
    gpg->cmd.cb_data = tmp;
    gpg->cmd.fnc = fnc;
    gpg->cmd.fnc_value = fnc_value;
    gpg->cmd.used = 1;
    return 0;
}


static void
free_argv ( char **argv )
{
    int i;

    for (i=0; argv[i]; i++ )
        safe_free (argv[i]);
    safe_free (argv);
}

static void
free_fd_data_map ( struct fd_data_map_s *fd_data_map )
{
    int i;

    if ( !fd_data_map )
        return;

    for (i=0; fd_data_map[i].data; i++ ) {
        if ( fd_data_map[i].fd != -1 )
            _gpgme_io_close (fd_data_map[i].fd);
        if ( fd_data_map[i].peer_fd != -1 )
            _gpgme_io_close (fd_data_map[i].peer_fd);
        /* don't release data because this is only a reference */
    }
    safe_free (fd_data_map);
}


static gpgme_error_t
build_argv (_gpg_object_t gpg)
{
    struct arg_and_data_s *a;
    struct fd_data_map_s *fd_data_map;
    size_t datac=0, argc=0;  
    char **argv;
    int need_special = 0;
    int use_agent = !!getenv ("GPG_AGENT_INFO");	
    const char *optfile = _gpgme_get_gpg_optfile (0);
    const char *pubring = _gpgme_get_pubring();
    const char *secring = _gpgme_get_secring();
       
    if (gpg->argv) {
        free_argv (gpg->argv);
        gpg->argv = NULL;
    }
    if (gpg->fd_data_map) {
        free_fd_data_map (gpg->fd_data_map);
        gpg->fd_data_map = NULL;
    }

    argc++; /* for argv[0] */
    for ( a=gpg->arglist; a; a = a->next ) {
        argc++;
        if (a->data) {
            /*fprintf (stderr, "build_argv: data\n" );*/
            datac++;
            if ( a->dup_to == -1 && !a->print_fd )
                need_special = 1;
        }
        else {
            /*   fprintf (stderr, "build_argv: arg=`%s'\n", a->arg );*/
        }
    }
    if ( need_special )
        argc++;
    if (use_agent)
        argc++;
	if (optfile)
		argc += 2;
    if (!gpg->cmd.used)
        argc++;
    if (pubring)
        argc += 2;
    if (secring)
        argc += 2;

    argv = calloc ( argc+1, sizeof *argv );
    if (!argv)
        return mk_error (Out_Of_Core);
    fd_data_map = calloc ( datac+1, sizeof *fd_data_map );
    if (!fd_data_map) {
        free_argv (argv);
        return mk_error (Out_Of_Core);
    }

    argc = datac = 0;
    argv[argc] = strdup ( "gpg" ); /* argv[0] */
    if (!argv[argc]) {
        safe_free (fd_data_map);
        free_argv (argv);
        return mk_error (Out_Of_Core);
    }
    argc++;
    if (need_special) {
        argv[argc] = strdup ( "--enable-special-filenames" );
        if (!argv[argc]) {
            safe_free (fd_data_map);
            free_argv (argv);
            return mk_error (Out_Of_Core);
        }
        argc++;
    }
    if (use_agent) {
        argv[argc] = strdup ("--use-agent");
        if( !argv[argc] ) {
            safe_free (fd_data_map);
            free_argv (argv);
            return mk_error (Out_Of_Core);
        }
        argc++;
    }
    if( optfile ) {
	char *p;
	
	argv[argc] = strdup ("--options");
	if( !argv[argc] ) {
	    safe_free(fd_data_map);
	    free_argv(argv);
	    return mk_error(Out_Of_Core);
	}
	argc++;
	p = malloc (strlen (optfile)+3);
	if (!p)
	{
	    safe_free (fd_data_map);
	    free_argv (argv);
	    return mk_error (Out_Of_Core);
	}
	strcpy (p+1, optfile);
	p[0] = '"';
	p[strlen(optfile)+1] = '"';
	p[strlen(optfile)+2] = '\0';
	argv[argc] = p;
	if (!argv[argc]) {
	    safe_free(fd_data_map);
	    free_argv(argv);
	    return mk_error(Out_Of_Core);
	}
	argc++;
    }
    if( pubring ) {
	char *p;
        int len = strlen(pubring);

	argv[argc] = strdup ("--keyring");
	if( !argv[argc] ) {
	    safe_free(fd_data_map);
	    free_argv(argv);
	    return mk_error(Out_Of_Core);
	}
	argc++;
	p = malloc (len+3);
	if (!p)
	{
	    safe_free (fd_data_map);
	    free_argv (argv);
	    return mk_error (Out_Of_Core);
	}
	strcpy (p+1, pubring);
	p[0] = '"';
	p[len+1] = '"';
	p[len+2] = '\0';
	argv[argc] = p;
	if (!argv[argc]) {
	    safe_free(fd_data_map);
	    free_argv(argv);
	    return mk_error(Out_Of_Core);
	}
	argc++;
    }
    if( secring ) {
	char *p;
        int len = strlen(secring);

	argv[argc] = strdup ("--secret-keyring");
	if( !argv[argc] ) {
	    safe_free(fd_data_map);
	    free_argv(argv);
	    return mk_error(Out_Of_Core);
	}
	argc++;
	p = malloc (len+3);
	if (!p)
	{
	    safe_free (fd_data_map);
	    free_argv (argv);
	    return mk_error (Out_Of_Core);
	}
	strcpy (p+1, secring);
	p[0] = '"';
	p[len+1] = '"';
	p[len+2] = '\0';
	argv[argc] = p;
	if (!argv[argc]) {
	    safe_free(fd_data_map);
	    free_argv(argv);
	    return mk_error(Out_Of_Core);
	}
	argc++;
    }
    if (!gpg->cmd.used) {
        argv[argc] = strdup ( "--batch" );
        if (!argv[argc]) {
            safe_free (fd_data_map);
            free_argv (argv);
            return mk_error (Out_Of_Core);
        }
        argc++;
    }
    for ( a=gpg->arglist; a; a = a->next ) {
        if ( a->data ) {
            switch ( _gpgme_data_get_mode (a->data) ) {
              case GPGME_DATA_MODE_NONE:
              case GPGME_DATA_MODE_INOUT:
                safe_free (fd_data_map);
                free_argv (argv);
                return mk_error (Invalid_Mode);
              case GPGME_DATA_MODE_IN:
                /* create a pipe to read from gpg */
                fd_data_map[datac].inbound = 1;
                break;
              case GPGME_DATA_MODE_OUT:
                /* create a pipe to pass it down to gpg */
                fd_data_map[datac].inbound = 0;
                break;
            }

            switch ( gpgme_data_get_type (a->data) ) {
              case GPGME_DATA_TYPE_NONE:
                if ( fd_data_map[datac].inbound )
                    break;  /* allowed */
                safe_free (fd_data_map);
                free_argv (argv);
                return mk_error (Invalid_Type);
              case GPGME_DATA_TYPE_MEM:
              case GPGME_DATA_TYPE_CB:
                break;
              case GPGME_DATA_TYPE_FD:
              case GPGME_DATA_TYPE_FILE:
                safe_free (fd_data_map);
                free_argv (argv);
                return mk_error (Not_Implemented);
            }
  
            /* create a pipe */
            {   
                int fds[2];
                
                if (_gpgme_io_pipe (fds, fd_data_map[datac].inbound?1:0 )
                    == -1) {
                    safe_free (fd_data_map);
                    free_argv (argv);
                    return mk_error (Pipe_Error);
                }
                if ( _gpgme_io_set_close_notify (fds[0],
                                                 close_notify_handler, gpg)
                     || _gpgme_io_set_close_notify (fds[1],
                                                    close_notify_handler,
                                                    gpg)) {
                    return mk_error (General_Error);
                }
                /* if the data_type is FD, we have to do a dup2 here */
                if (fd_data_map[datac].inbound) {
                    fd_data_map[datac].fd       = fds[0];
                    fd_data_map[datac].peer_fd  = fds[1];
                }
                else {
                    fd_data_map[datac].fd       = fds[1];
                    fd_data_map[datac].peer_fd  = fds[0];
                }
            }

            /* Hack to get hands on the fd later */
            if ( gpg->cmd.used && gpg->cmd.cb_data == a->data ) {
                assert (gpg->cmd.fd == -1);
                gpg->cmd.fd = fd_data_map[datac].fd;
            }

            fd_data_map[datac].data = a->data;
            fd_data_map[datac].dup_to = a->dup_to;
            if ( a->dup_to == -1 ) {
                argv[argc] = malloc ( 25 );
                if (!argv[argc]) {
                    safe_free (fd_data_map);
                    free_argv (argv);
                    return mk_error (Out_Of_Core);
                }
                sprintf ( argv[argc], 
                          a->print_fd? "%d" : "-&%d",
                          fd_data_map[datac].peer_fd );
                argc++;
            }
            datac++;
        }
        else {
            argv[argc] = strdup ( a->arg );
            if (!argv[argc]) {
                safe_free (fd_data_map);
                free_argv (argv);
                return mk_error (Out_Of_Core);
            }
            argc++;
        }
    }

    gpg->argv = argv;
    gpg->fd_data_map = fd_data_map;
    return 0;
}


ulong
_gpgme_gpg_getpid( _gpg_object_t gpg )
{
    assert( gpg );
    return gpg->pid;
}


void
_gpgme_gpg_set_logging_handler( _gpg_object_t gpg, void * cb_val )
{
    assert( gpg );
    gpg->logging.fnc_value = cb_val;
    gpg->logging.enabled = 1;
}


gpgme_error_t
_gpgme_gpg_spawn( _gpg_object_t gpg, void *opaque )
{
    int rc;
    int i, n;
    int pid;
    struct spawn_fd_item_s *fd_child_list, *fd_parent_list;
    const char *gpg_path = _gpgme_get_gpg_path (0);

    if ( !gpg )
        return mk_error (Invalid_Value);

    /* Kludge, so that we don't need to check the return code of
     * all the gpgme_gpg_add_arg().  we bail out here instead */
    if ( gpg->arg_error )
        return mk_error (Out_Of_Core);

    rc = build_argv ( gpg );
    if ( rc )
        return rc;

    n = 4; /* status fd, 2*colon_fd and end of list */
    for (i=0; gpg->fd_data_map[i].data; i++ ) 
        n += 2;
    fd_child_list = calloc ( n+n, sizeof *fd_child_list );
    if (!fd_child_list)
        return mk_error (Out_Of_Core);
    fd_parent_list = fd_child_list + n;

    /* build the fd list for the child */
    n=0;
    fd_child_list[n].fd = gpg->status.fd[0]; 
    fd_child_list[n].dup_to = -1;
    n++;
    if( gpg->logging.enabled ) {
	fd_child_list[n].fd = gpg->logging.fd[0];
	fd_child_list[n].dup_to = -1;
	n++;
    }
    if ( gpg->colon.fnc ) {
        fd_child_list[n].fd = gpg->colon.fd[0];
        fd_child_list[n].dup_to = -1;
        n++;
        fd_child_list[n].fd = gpg->colon.fd[1]; 
        fd_child_list[n].dup_to = 1; /* dup to stdout */
        n++;
    }
    for (i=0; gpg->fd_data_map[i].data; i++ ) {
        fd_child_list[n].fd = gpg->fd_data_map[i].fd;
        fd_child_list[n].dup_to = -1;
        n++;
        if (gpg->fd_data_map[i].dup_to != -1) {
            fd_child_list[n].fd = gpg->fd_data_map[i].peer_fd;
            fd_child_list[n].dup_to = gpg->fd_data_map[i].dup_to;
            n++;
        }
    }
    fd_child_list[n].fd = -1;
    fd_child_list[n].dup_to = -1;

    /* build the fd list for the parent */
    n=0;
    if ( gpg->status.fd[1] != -1 ) {
        fd_parent_list[n].fd = gpg->status.fd[1];
        fd_parent_list[n].dup_to = -1;
        n++;
        gpg->status.fd[1] = -1;
    }
    if( gpg->logging.fd[1] != -1 ) {
	fd_parent_list[n].fd = gpg->logging.fd[1];
	fd_parent_list[n].dup_to = -1;
	n++;
	gpg->logging.fd[1] = -1;
    }
    if ( gpg->colon.fd[1] != -1 ) {
        fd_parent_list[n].fd = gpg->colon.fd[1];
        fd_parent_list[n].dup_to = -1;
        n++;
        gpg->colon.fd[1] = -1;
    }
    for (i=0; gpg->fd_data_map[i].data; i++ ) {
        fd_parent_list[n].fd = gpg->fd_data_map[i].peer_fd;
        fd_parent_list[n].dup_to = -1;
        n++;
        gpg->fd_data_map[i].peer_fd = -1;
    }        
    fd_parent_list[n].fd = -1;
    fd_parent_list[n].dup_to = -1;


    if (gpg->path)
	gpg_path = gpg->path;
    pid = _gpgme_io_spawn (gpg_path, gpg->argv, fd_child_list, fd_parent_list);
    safe_free (fd_child_list);
    if (pid == -1)
        return mk_error (Exec_Error);

    gpg->pid = pid;

    /*_gpgme_register_term_handler ( closure, closure_value, pid );*/

    if( _gpgme_register_pipe_handler( opaque, gpg_status_handler,
				      gpg, pid, gpg->status.fd[0], 1 ) ) {
	/* FIXME: kill the child */
	return mk_error( General_Error );
    }
    if( gpg->logging.enabled && 
	_gpgme_register_pipe_handler( opaque, gpg_logging_handler,
				      gpg, pid, gpg->logging.fd[0], 1 ) ) {
        /* FIXME: kill the child */
        return mk_error( General_Error );
    }

    if( gpg->colon.fnc ) {
        assert( gpg->colon.fd[0] != -1 );
        if ( _gpgme_register_pipe_handler ( opaque, gpg_colon_line_handler,
                                            gpg, pid, gpg->colon.fd[0], 1 ) ) {
            /* FIXME: kill the child */
            return mk_error (General_Error);
            
        }
    }

    for (i=0; gpg->fd_data_map[i].data; i++ ) {
        /* Due to problems with select and write we set outbound pipes
         * to non-blocking */
        if (!gpg->fd_data_map[i].inbound)
            _gpgme_io_set_nonblocking (gpg->fd_data_map[i].fd);

        if ( _gpgme_register_pipe_handler (
                 opaque, 
                 gpg->fd_data_map[i].inbound?
                       gpg_inbound_handler:gpg_outbound_handler,
                 gpg->fd_data_map[i].data,
                 pid, gpg->fd_data_map[i].fd,
                 gpg->fd_data_map[i].inbound )
           ) {
            /* FIXME: kill the child */
            return mk_error (General_Error);
        }
    }

    if (gpg->cmd.used)
        _gpgme_freeze_fd (gpg->cmd.fd);

    /* fixme: check what data we can release here */
    
    gpg->running = 1;
    return 0;
}


static int
gpg_inbound_handler ( void *opaque, int pid, int fd )
{
    gpgme_data_t dh = opaque;
    gpgme_error_t err;
    int nread;
    char buf[200];

    assert ( _gpgme_data_get_mode (dh) == GPGME_DATA_MODE_IN );

    nread = _gpgme_io_read (fd, buf, 200 );
    if ( nread < 0 ) {
        DEBUG3 ("read_mem_data: read failed on fd %d (n=%d): %s",
                 fd, nread, strerror (errno) );
        return 1;
    }
    else if (!nread)
        return 1; /* eof */

    /* We could improve this with a gpgme_data_t function which takes
     * the read function or provides a memory area for writing to it.
     */
    
    err = _gpgme_data_append ( dh, buf, nread );
    if ( err ) {
        DEBUG1 ("_gpgme_append_data failed: %s\n",
                 gpgme_strerror(err));
        /* Fixme: we should close the pipe or read it to /dev/null in
         * this case. Returnin EOF is not sufficient */
        return 1;
    }

    return 0;
}


static int
write_mem_data ( gpgme_data_t dh, int fd )
{
    size_t nbytes;
    int  nwritten; 

    nbytes = dh->len - dh->readpos;
    if ( !nbytes ) {
        _gpgme_io_close (fd);
        return 1;
    }
    
    /* FIXME: Arggg, the pipe blocks on large write request, although
     * select told us that it is okay to write - need to figure out
     * why this happens?  Stevens says nothing about this problem (or
     * is it my Linux kernel 2.4.0test1)
     * To avoid that we have set the pipe to nonblocking.
     */

    nwritten = _gpgme_io_write ( fd, dh->data+dh->readpos, nbytes );
    if (nwritten == -1 && errno == EAGAIN )
        return 0;
    if ( nwritten < 1 ) {
        DEBUG3 ("write_mem_data(%d): write failed (n=%d): %s",
                fd, nwritten, strerror (errno) );
        _gpgme_io_close (fd);
        return 1;
    }

    dh->readpos += nwritten;
    return 0;
}

static int
write_cb_data( gpgme_data_t dh, int fd )
{
    size_t nbytes;
    int  err, nwritten; 
    char buffer[512];

    err = gpgme_data_read ( dh, buffer, DIM(buffer), &nbytes );
    if (err == GPGME_EOF) {
        _gpgme_io_close (fd);
        return 1;
    }
    
    nwritten = _gpgme_io_write ( fd, buffer, nbytes );
    if (nwritten == -1 && errno == EAGAIN )
        return 0;
    if ( nwritten < 1 ) {
        DEBUG3 ("write_cb_data(%d): write failed (n=%d): %s",
                fd, nwritten, strerror (errno) );
        _gpgme_io_close (fd);
        return 1;
    }

    if ( (unsigned)nwritten < nbytes ) {
        /* ugly, ugly: It does currently only for for MEM type data */
        if ( _gpgme_data_unread (dh, buffer + nwritten, nbytes - nwritten ) )
            DEBUG1 ("wite_cb_data: unread of %d bytes failed\n",
                     nbytes - nwritten );
        _gpgme_io_close (fd);
        return 1;
    }

    return 0;
}


static int
gpg_outbound_handler ( void *opaque, int pid, int fd )
{
    gpgme_data_t dh = opaque;

    assert ( _gpgme_data_get_mode (dh) == GPGME_DATA_MODE_OUT );
    switch ( gpgme_data_get_type (dh) ) {
      case GPGME_DATA_TYPE_MEM:
        if ( write_mem_data ( dh, fd ) )
            return 1; /* ready */
        break;
      case GPGME_DATA_TYPE_CB:
        if (write_cb_data (dh, fd))
            return 1; /* ready */
        break;
      default:
        assert (0);
    }

    return 0;
}



static int
gpg_logging_handler( void * opaque, int pid, int fd )
{
    _gpg_object_t gpg = opaque;
    int rc = 0;

    assert( fd == gpg->logging.fd[0] );
    rc = read_logging( gpg );
    if( rc ) {
	DEBUG1( "gpg_handler: read_logging problem %d\n - stop", rc );
	return 1;
    }
    return gpg->logging.eof;
}


static int
gpg_status_handler ( void *opaque, int pid, int fd )
{
    _gpg_object_t gpg = opaque;
    int rc = 0;

    assert ( fd == gpg->status.fd[0] );
    rc = read_status ( gpg );
    if ( rc ) {
        DEBUG1 ("gpg_handler: read_status problem %d\n - stop", rc);
        return 1;
    }

    return gpg->status.eof;
}



static int
status_cmp (const void *ap, const void *bp)
{
    const struct status_table_s *a = ap;
    const struct status_table_s *b = bp;

    return strcmp (a->name, b->name);
}


static gpgme_error_t
read_logging( _gpg_object_t gpg )
{
    gpgme_ctx_t c;
    int nread;

    c = gpg->logging.fnc_value;
    if( !c )
	return mk_error( Invalid_Value );
    if( !c->use_logging )
	return 0;

    nread = _gpgme_io_read( gpg->logging.fd[0], 
			    gpg->logging.buffer, 
			    gpg->logging.bufsize );
    DEBUG1( "read_logging: got %d bytes", nread );
    if( nread == -1 )
	return mk_error( Read_Error );

    if( !c->logging )
	gpgme_data_new( &c->logging );

    if( !nread ) {
	gpg->logging.eof = 1;
	return 0;
    }
    
    return gpgme_data_write( c->logging, gpg->logging.buffer, nread );
}


/*
 * Handle the status output of GnuPG.  This function does read entire
 * lines and passes them as C strings to the callback function (we can
 * use C Strings because the status output is always UTF-8 encoded).
 * Of course we have to buffer the lines to cope with long lines
 * e.g. with a large user ID.  Note: We can optimize this to only cope
 * with status line code we know about and skip all other stuff
 * without buffering (i.e. without extending the buffer).  */
static gpgme_error_t
read_status ( _gpg_object_t gpg )
{
    char *p;
    int nread;
    size_t bufsize = gpg->status.bufsize; 
    char *buffer = gpg->status.buffer;
    size_t readpos = gpg->status.readpos; 

    assert (buffer);
    if (bufsize - readpos < 256) { 
        /* need more room for the read */
        bufsize += 1024;
        buffer = realloc (buffer, bufsize);
        if ( !buffer ) 
            return mk_error (Out_Of_Core);
    }
    

    nread = _gpgme_io_read ( gpg->status.fd[0],
                             buffer+readpos, bufsize-readpos );
    if (nread == -1)
        return mk_error(Read_Error);

    if (!nread) {
        gpg->status.eof = 1;
        if (gpg->status.fnc)
            gpg->status.fnc ( gpg->status.fnc_value, STATUS_EOF, "" );
        return 0;
    }

    while (nread > 0) {
        for (p = buffer + readpos; nread; nread--, p++) {
            if ( *p == '\n' ) {
                /* (we require that the last line is terminated by a LF) */
                *p = 0;
                /*DEBUG1("read_status: `%s'\n", buffer);*/
                if (!strncmp (buffer, "[GNUPG:] ", 9 )
                    && buffer[9] >= 'A' && buffer[9] <= 'Z' ) {
                    struct status_table_s t, *r;
                    char *rest;

                    rest = strchr (buffer+9, ' ');
                    if ( !rest )
                        rest = p; /* set to an empty string */
                    else
                        *rest++ = 0;

                    t.name = buffer+9;
                    /* (the status table as one extra element) */
                    r = bsearch ( &t, status_table, DIM(status_table)-1,
                                  sizeof t, status_cmp );
                    if ( r ) {
                        if ( gpg->cmd.used
                             && ( r->code == STATUS_GET_BOOL
                                  || r->code == STATUS_GET_LINE
                                  || r->code == STATUS_GET_HIDDEN )) {
                            gpg->cmd.code = r->code;
                            safe_free (gpg->cmd.keyword);
                            gpg->cmd.keyword = strdup (rest);
                            if ( !gpg->cmd.keyword )
                                return mk_error (Out_Of_Core);
                            /* this should be the last thing we have received
                             * and the next thing will be that the command
                             * handler does its action */
                            if ( nread > 1 )
                                DEBUG0 ("ERROR, unexpected data in read_status");
                            _gpgme_thaw_fd (gpg->cmd.fd);
                        }
                        else if ( gpg->status.fnc ) {
                            gpg->status.fnc( gpg->status.fnc_value, 
				             r->code, rest );
                        }
                    
                        if ( r->code == STATUS_END_STREAM ) {
                            if ( gpg->cmd.used )
                                _gpgme_freeze_fd ( gpg->cmd.fd );
                        }
                    }
                }
                /* To reuse the buffer for the next line we have to
                 * shift the remaining data to the buffer start and
                 * restart the loop Hmmm: We can optimize this
                 * function by looking forward in the buffer to see
                 * whether a second complete line is available and in
                 * this case avoid the memmove for this line.  */
                nread--; p++;
                if (nread)
                    memmove (buffer, p, nread);
                readpos = 0;
                break; /* the for loop */
            }
            else
                readpos++;
        }
    } 

    /* Update the gpg object.  */
    gpg->status.bufsize = bufsize;
    gpg->status.buffer = buffer;
    gpg->status.readpos = readpos;
    return 0;
}


/*
 * This colonline handler thing is not the clean way to do it.
 * It might be better to enhance the gpgme_data_t object to act as
 * a wrapper for a callback.  Same goes for the status thing.
 * For now we use this thing here becuase it is easier to implement.
 */
static int
gpg_colon_line_handler ( void *opaque, int pid, int fd )
{
    _gpg_object_t gpg = opaque;
    gpgme_error_t rc = 0;

    assert ( fd == gpg->colon.fd[0] );
    rc = read_colon_line ( gpg );
    if ( rc ) {
        DEBUG1 ("gpg_colon_line_handler: "
                 "read problem %d\n - stop", rc);
        return 1;
    }

    return gpg->colon.eof;
}

static gpgme_error_t
read_colon_line ( _gpg_object_t gpg )
{
    char *p;
    int nread;
    size_t bufsize = gpg->colon.bufsize; 
    char *buffer = gpg->colon.buffer;
    size_t readpos = gpg->colon.readpos; 

    assert (buffer);
    if (bufsize - readpos < 256) { 
        /* need more room for the read */
        bufsize += 1024;
        buffer = realloc (buffer, bufsize);
        if ( !buffer ) 
            return mk_error (Out_Of_Core);
    }
    

    nread = _gpgme_io_read ( gpg->colon.fd[0],
                             buffer+readpos, bufsize-readpos );
    if (nread == -1)
        return mk_error(Read_Error);

    if (!nread) {
        gpg->colon.eof = 1;
        assert (gpg->colon.fnc);
        gpg->colon.fnc ( gpg->colon.fnc_value, NULL );
        return 0;
    }

    while (nread > 0) {
        for (p = buffer + readpos; nread; nread--, p++) {
            if ( *p == '\n' ) {
                /* (we require that the last line is terminated by a
                 * LF) and we skip empty lines.  Note: we use UTF8
                 * encoding and escaping of special characters
                 * We require at least one colon to cope with
                 * some other printed information.
                 */
                *p = 0;
                if ( gpg->colon.simple
                     || (*buffer && strchr (buffer, ':')) ) {
                    assert (gpg->colon.fnc);
                    gpg->colon.fnc ( gpg->colon.fnc_value, buffer );
                }
            
                /* To reuse the buffer for the next line we have to
                 * shift the remaining data to the buffer start and
                 * restart the loop Hmmm: We can optimize this
                 * function by looking forward in the buffer to see
                 * whether a second complete line is available and in
                 * this case avoid the memmove for this line.  */
                nread--; p++;
                if (nread)
                    memmove (buffer, p, nread);
                readpos = 0;
                break; /* the for loop */
            }
            else
                readpos++;
        }
    } 
    
    /* Update the gpg object.  */
    gpg->colon.bufsize = bufsize;
    gpg->colon.buffer  = buffer;
    gpg->colon.readpos = readpos;
    return 0;
}


/* 
 * Here we handle --command-fd.  This works closely together with
 * the status handler.  
 */

static int
command_cb ( void *opaque, char *buffer, size_t length, size_t *nread )
{
    _gpg_object_t gpg = opaque;
    const char *value;
    size_t value_len;

    DEBUG0 ("command_cb: enter\n");
    assert (gpg->cmd.used);
    if ( !buffer || !length || !nread )
        return 0; /* those values are reserved for extensions */
    *nread =0;
    if( !gpg->cmd.code ) {
        DEBUG0 ("command_cb: no code\n");
        return -1;
    }
    
    if( !gpg->cmd.fnc ) {
        DEBUG0 ("command_cb: no user cb\n");
        return -1;
    }

    value = gpg->cmd.fnc ( gpg->cmd.fnc_value, 
                           gpg->cmd.code, gpg->cmd.keyword );
    if( !value ) {
        DEBUG0( "command_cb: no data from user cb\n" );
        gpg->cmd.fnc ( gpg->cmd.fnc_value, 0, value);
        return -1;
    }

    value_len = strlen (value);
    if ( value_len+1 > length ) {
        DEBUG0( "command_cb: too much data from user cb\n" );
        gpg->cmd.fnc ( gpg->cmd.fnc_value, 0, value);
        return -1;
    }

    memcpy ( buffer, value, value_len );
    if ( !value_len || (value_len && value[value_len-1] != '\n') ) 
        buffer[value_len++] = '\n';
    *nread = value_len;
    
    gpg->cmd.fnc( gpg->cmd.fnc_value, 0, value );
    gpg->cmd.code = 0;
    /* and sleep again until read_status will wake us up again */
    _gpgme_freeze_fd ( gpg->cmd.fd );
    return 0;
}




