/*
 * Debugging library routines
 */
#if defined(ENABLE_LIBYAHOO_DEBUG)

#if defined(HAVE_CONFIG_H)
#include "config.h"
#else
#define HAVE_STDARG_H
#define HAVE_STRING_H
#endif

#include "libyahoo-debug.h"

#include <stdlib.h>
#include <stdio.h>
#if defined(HAVE_STDARG_H)
#include <stdarg.h>
#endif

#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif

static FILE *debugfile = NULL;
static char *yahoo_dbg_EnabledKeys[128] = { NULL };	/* should be more than enough */
static int yahoo_dbg_MaxEnabledKey = -1;

/* these routines currently do a really inneficient linear search, 
   but it's not like they need to be really fast - they are just for
   debugging purposes */

int yahoo_dbg_Open(char *file)
{
	if (debugfile && debugfile != stdout)
	{
		fclose(debugfile);
		debugfile = NULL;
	}

	if (!file)
	{
		fprintf(stderr, "yahoo_dbg_Open Error: File is null, using stdout.\n");
		debugfile = stdout;
		return 1;
	}

	if ((debugfile = fopen(file, "a")) == NULL)
	{
		fprintf(stderr, "yahoo_dbg_Open Error: Couldn't open output file.\n");
		fprintf(stderr, "                Falling back to stdout.\n");
		debugfile = stdout;
		return 1;
	}

	return 0;
}

int yahoo_dbg_Disable(char *key)
{
	int i;

	if (!key)
	{
		fprintf(stderr, "yahoo_dbg_Disable Error: key is null.\n");
		return 1;
	}
	for (i = 0; i <= yahoo_dbg_MaxEnabledKey; i++)
	{
		if (yahoo_dbg_EnabledKeys[i] && !strcmp(yahoo_dbg_EnabledKeys[i], key))
		{
			free(yahoo_dbg_EnabledKeys[i]);
			yahoo_dbg_EnabledKeys[i] = NULL;
			return 0;
		}
	}
	return 1;
}

int yahoo_dbg_Enable(char *key)
{
	int i;

	if (!key)
	{
		fprintf(stderr, "yahoo_dbg_Enable Error: key is null.\n");
		return 1;
	}
	for (i = 0; i <= yahoo_dbg_MaxEnabledKey; i++)
	{
		if (yahoo_dbg_EnabledKeys[i] && !strcmp(yahoo_dbg_EnabledKeys[i], key))
		{
			return 0;			/* already enabled */
		}
	}
	yahoo_dbg_EnabledKeys[yahoo_dbg_MaxEnabledKey + 1] = NULL;
	for (i = 0; i <= yahoo_dbg_MaxEnabledKey + 1; i++)
	{
		if (!yahoo_dbg_EnabledKeys[i])
		{
			yahoo_dbg_EnabledKeys[i] = strdup(key);
			if (i > yahoo_dbg_MaxEnabledKey)
			{
				yahoo_dbg_MaxEnabledKey = i;
			}
			return 0;			/* already enabled */
		}
	}
	return 0;
}

int yahoo_dbg_Print(char *key, char *format, ...)
{
	va_list args;

	if (!key)
	{
		fprintf(stderr, "yahoo_dbg_Print Error: key is null.\n");
		return 1;
	}
	if (!format)
	{
		fprintf(stderr, "yahoo_dbg_Print Error: format is null.\n");
		return 1;
	}

	if (!debugfile)
	{
		debugfile = stdout;
	}

	if (yahoo_dbg_IsEnabled(key) || yahoo_dbg_IsEnabled("all"))
	{
		va_start(args, format);
		vfprintf(debugfile, format, args);
		va_end(args);

		fflush(debugfile);
	}
	return 0;
}

int yahoo_dbg_Close(void)
{
	if (debugfile && debugfile != stdout)
	{
		fclose(debugfile);
		debugfile = NULL;
	}
	return 0;
}

int yahoo_dbg_IsEnabled(char *key)
{
	int i;

	for (i = 0; i <= yahoo_dbg_MaxEnabledKey; i++)
	{
		if (!strcmp(yahoo_dbg_EnabledKeys[i], key))
		{
			return 1;			/* is enabled */
		}
	}
	return 0;
}

#else

/* Debugging is not enabled, so do an inline routines that do nothing */
/* Hopefully the compiler will completely optimize these routines away */

inline int yahoo_dbg_Open(char *file)
{
	return 0;
}

inline int yahoo_dbg_Disable(char *key)
{
	return 0;
}

inline int yahoo_dbg_Enable(char *key)
{
	return 0;
}

inline int yahoo_dbg_Print(char *key, char *format, ...)
{
	return 0;
}

inline int yahoo_dbg_Close(void)
{
	return 0;
}

inline int yahoo_dbg_IsEnabled(char *key)
{
	return 0;
}

#endif
