/* This is part of Ari's Yahoo Client.
 * This file and Ari's Yahoo Client are copyright (C) 1999 Ari Johnson.
 * For more copyright and licensing information, see the file
 * 'COPYING', which should have been distributed with Ari's Yahoo Client.
 *
 * Yahoo Messenger interface code provided by yahoolib, as provided
 * with GTKYahoo.
 */

#include "config.h"

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>

#include "libyahoo.h"
#include "libyahoo-proto.h"

struct statusrec {
  char *id;
  int status;
  int in_pager;
  int in_chat;
  struct statusrec *next;
};

int debug_packets = 0;
int quit_flag;
struct statusrec *statushead = NULL;
struct yahoo_options options;
struct yahoo_context *context;
char *username;
char *password;
char *cur_target = NULL;
int cur_status = YAHOO_STATUS_AVAILABLE;

void main_loop(void);
void process_input(void);
void handle_io(void);

int main(int argc, char *argv[]) {
  int i;
  struct yahoo_buddy *tmpbuddy;
  struct statusrec *tmprec;
  char *id;

  if(argc > 2) {
    fprintf(stderr, "Usage:  %s [<username>]\n", argv[0]);
    return 1;
  }

  if(argc == 1) {
    username = (char *) malloc(100);
    printf("Yahoo ID: ");
    fflush(stdout);
    scanf("%s", username);
  } else {
    username = (char *) malloc(strlen(argv[1]) + 1);
    strcpy(username, argv[1]);
  }

  password = getpass("Password: ");

  options.connect_mode = YAHOO_CONNECT_NORMAL;

  context = yahoo_init(username, password, &options);

  fflush(stdout);
  if(!yahoo_connect(context)) {
    fprintf(stderr, "Connection failed.\n");
    return 1;
  }

  yahoo_get_config(context);

  yahoo_cmd_logon(context, cur_status);

  i = 0;
  while(context->buddies && context->buddies[i])
    i++;
  i--;
  while(i >= 0) {
    tmpbuddy = context->buddies[i];
    id = tmpbuddy->id;

    tmprec = (struct statusrec *) malloc(sizeof(struct statusrec));
    tmprec->id = id;
    tmprec->status = cur_status;
    tmprec->in_pager = 0;
    tmprec->in_chat = 0;
    tmprec->next = statushead;
    statushead = tmprec;
    
    i--;
  }

  main_loop();

  printf("\n\n");
  printf("Thank you for using Ari's Yahoo Client.\n");
  printf("Ari's Yahoo Client is copyright (C) 1999 Ari Johnson.\n\n");
  printf("Yahoo Messenger interface provided by yahoolib,\n");
  printf("as distributed with GTKYahoo.\n");

  free(username);
  free(password);
  return 0;
}

struct statusrec *get_statusrec(char *id) {
  struct statusrec *ptr;

  ptr = statushead;
  while(ptr && strcasecmp(id, ptr->id))
    ptr = ptr->next;

  return ptr;
}

int get_in_pager(char *id) {
  struct statusrec *rec;

  rec = get_statusrec(id);
  if(!rec)
    return 0;

  return rec->in_pager;
}

int get_in_chat(char *id) {
  struct statusrec *rec;

  rec = get_statusrec(id);
  if(!rec)
    return 0;

  return rec->in_chat;
}

int get_status(char *id) {
  struct statusrec *rec;

  rec = get_statusrec(id);
  if(!rec)
    return YAHOO_STATUS_AVAILABLE;

  return rec->status;
}

void set_in_pager(char *id, int in) {
  struct statusrec *rec;

  rec = get_statusrec(id);
  if(!rec)
    return;

  rec->in_pager = in;
}

void set_in_chat(char *id, int in) {
  struct statusrec *rec;

  rec = get_statusrec(id);
  if(!rec)
    return;

  rec->in_chat = in;
}

void set_status(char *id, int status) {
  struct statusrec *rec;

  rec = get_statusrec(id);
  if(!rec)
    return;

  rec->status = status;
}

void show_userlist(void) {
  int i;
  struct yahoo_buddy *tmpbuddy;
  char *id;
  struct statusrec *rec;
  int status, in_pager, in_chat;

  printf("\n");

  i = 0;
  while(context->buddies && context->buddies[i]) {
    tmpbuddy = context->buddies[i];
    id = tmpbuddy->id;

    status = get_status(id);
    in_pager = get_in_pager(id);
    in_chat = get_in_chat(id);

    printf("%c  %s", (in_pager || in_chat) ? '*' : ' ', id);
    if(status != YAHOO_STATUS_AVAILABLE && (in_pager || in_chat))
      printf(" [%s]\n", yahoo_get_status_string(status));
    else
      printf("\n");
    i++;
  }
}

int do_add(char *id) {
  int i;
  struct yahoo_buddy *tmpbuddy;
  char *group;

  group = NULL;
  if(context->buddies && context->buddies[0]) {
    tmpbuddy = context->buddies[0];
    group = tmpbuddy->group;
  }


  if(group) {
    yahoo_add_buddy(context, id, username, group, "");
    return 0;
  }

  return 1;
}

int do_remove(char *id) {
  int i;
  struct yahoo_buddy *tmpbuddy;
  char *group;

  group = NULL;
  i = 0;
  while(context->buddies && context->buddies[i]) {
    tmpbuddy = context->buddies[i];
    if(!strcmp(tmpbuddy->id, id)) {
      group = tmpbuddy->group;
    }
    i++;
  }

  if(group) {
    yahoo_remove_buddy(context, id, username, group, "");
    return 0;
  }

  return 1;
}

void show_prompt(void) {
  if(cur_status != YAHOO_STATUS_AVAILABLE)
    printf("[%s] ", yahoo_get_status_string(cur_status));
  printf("%s:  ", cur_target ? cur_target : "<none>");
  fflush(stdout);
}

void main_loop(void) {
  fd_set inset;

  quit_flag = 0;

  printf("Connected.\n");

  show_prompt();

  while(!quit_flag) {
    FD_ZERO(&inset);
    FD_SET(fileno(stdin), &inset);
    FD_SET(context->sockfd, &inset);

    if(select(context->sockfd + 1, &inset, NULL, NULL, NULL) < 0) {
      perror("select");
      return;
    }

    if(FD_ISSET(fileno(stdin), &inset))
      process_input();
    if(FD_ISSET(context->sockfd, &inset))
      handle_io();
  }

  yahoo_cmd_logoff(context);
}

void process_command(char *cmd) {
  if(!strlen(cmd))
    return;
  if(cmd[0] == '/') {
    if(!strncasecmp(&cmd[1], "help", 4)) {
      printf("Ari's Yahoo Client Help\n");
      printf("--------------------------------------\n");
      printf("Commands:\n");
      printf("  /help         This screen.\n");
      printf("  /who          Display friends list.\n");
      printf("  /query <ID>   Select <ID> as target.\n");
      printf("  /query        Unselect target.\n");
      printf("  /status <#>   Set status to #.\n");
      printf("  /add <ID>     Add a friend.\n");
      printf("  /remove <ID>  Remove a friend.\n");
      printf("  /quit         Exit the program.\n");
      printf("--------------------------------------\n");
    } else if(!strncasecmp(&cmd[1], "who", 3)) {
      show_userlist();
    } else if(!strncasecmp(&cmd[1], "query", 5)) {
      if(cur_target)
        free(cur_target);
      cur_target = NULL;
      if(cmd[6] && cmd[7]) {
        cur_target = (char *) malloc(strlen(&cmd[7]) + 1);
        strncpy(cur_target, &cmd[7], strlen(&cmd[7]));
        cur_target[strlen(&cmd[7])] = '\0';
        printf("Target '%s' selected.\n", cur_target);
      } else {
        printf("Unselected target.\n");
      }
    } else if(!strncasecmp(&cmd[1], "status", 6)) {
      if(cmd[7] && cmd[8]) {
        int newstatus;
        newstatus = atoi(&cmd[8]);
        if(yahoo_get_status_string(newstatus)) {
          if(newstatus == YAHOO_STATUS_AVAILABLE)
            yahoo_cmd_set_back_mode(context, newstatus, NULL);
          else
            yahoo_cmd_set_away_mode(context, newstatus, NULL);
          printf("Changing status to %s.\n",
			yahoo_get_status_string(newstatus));
          cur_status = newstatus;
        } else {
          int i;
          printf("Invalid status number.\n");
          for(i = 0; i <= YAHOO_STATUS_IDLE; i++)
            if(yahoo_get_status_string(i))
              printf("%5d: %s\n", i, yahoo_get_status_string(i));
        }
      } else {
        int i;
        printf("You must specify a status number.\n");
        for(i = 0; i <= YAHOO_STATUS_IDLE; i++)
          if(yahoo_get_status_string(i))
            printf("%5d: %s\n", i, yahoo_get_status_string(i));
      }
    } else if(!strncasecmp(&cmd[1], "add", 3)) {
      if(cmd[4] && cmd[5]) {
        if(do_add(&cmd[5]))
          printf("Could not add friend.\n");
        else
          printf("Added `%s' to friends list.\n", &cmd[5]);
      } else {
        printf("You must give a Yahoo ID to add to your list.\n");
      }
    } else if(!strncasecmp(&cmd[1], "remove", 6)) {
      if(cmd[7] && cmd[8]) {
        if(do_remove(&cmd[8]))
          printf("Could not remove friend.\n");
        else
          printf("Removed `%s' from friends list.\n", &cmd[8]);
      } else {
        printf("You must give a Yahoo ID to remove from your list.\n");
      }
    } else if(!strncasecmp(&cmd[1], "quit", 4)) {
      quit_flag = 1;
    } else {
      printf("Invalid command.\n");
    }
  } else {
    if(!cur_target) {
      printf("No target selected.\n");
      return;
    }
    yahoo_cmd_msg(context, username, cur_target, cmd);
  }
}

void process_input(void) {
  char buff[1024];
  char *ptr;

  fgets(buff, 1024, stdin);
  ptr = (char *) strchr(buff, '\n');
  if(ptr)
    *ptr = '\0';
  ptr = (char *) strchr(buff, '\r');
  if(ptr)
    *ptr = '\0';
  process_command(buff);
  show_prompt();
}

void process_packet_status(struct yahoo_packet *pkt) {
  int i;

  if(pkt->service == YAHOO_SERVICE_LOGOFF &&
     !strcmp(pkt->active_id, username)) {
    printf("\nDisconnected.\n");
    quit_flag = 1;
  }

  for(i = 0; i < pkt->idstatus_count; i++) {
    struct yahoo_idstatus *rec;
    char *id;

    rec = pkt->idstatus[i];
    id = rec->id;

    set_in_pager(id, rec->in_pager);
    set_in_chat(id, rec->in_chat);
    if(pkt->service != YAHOO_SERVICE_CHATLOGOFF &&
       pkt->service != YAHOO_SERVICE_CHATLOGON) {
      set_status(id, rec->status);
    }

    if(pkt->service == YAHOO_SERVICE_LOGON)
      printf("\n%s has connected.\n", id);
    else if(pkt->service == YAHOO_SERVICE_LOGOFF)
      printf("\n%s has disconnected.\n", id);
    else
      printf("\n%s is now %s.\n", id, yahoo_get_status_string(get_status(id)));
    show_prompt();
  }
}

void process_packet_message(struct yahoo_packet *pkt) {
  if(pkt->msgtype == YAHOO_MSGTYPE_STATUS)
    set_status(pkt->msg_id, pkt->msg_status);

  if(pkt->msg) {
    if(pkt->msgtype == YAHOO_MSGTYPE_BOUNCE)
      printf("\n%s -> <Message not sent, user not online.>\n", pkt->msg_id);
    else
      printf("\n%s -> %s\n", pkt->msg_id, pkt->msg);
    show_prompt();
  }
}

void process_packet_ping(struct yahoo_packet *pkt) {
  printf("\nPing.\n");
}

void handle_io(void) {
  struct yahoo_rawpacket *rawpkt;
  struct yahoo_packet *pkt;

  yahoo_getdata(context);

  if(context->io_buf_curlen <= 103)
    return;

  while((rawpkt = yahoo_getpacket(context))) {
    pkt = yahoo_parsepacket(context, rawpkt);

    if(debug_packets) {
      printf("\nReceived packet:\n");
      printf("\tService = %s\n", yahoo_get_service_string(pkt->service));
      printf("\tReal ID = %s\n", pkt->real_id);
      printf("\tActive ID = %s\n", pkt->active_id);
      printf("\tConnection ID = %X\n", pkt->connection_id);
      printf("\tMagic ID = %X\n", pkt->magic_id);
      printf("\tUnknown Flag 1 = %X\n", pkt->unknown1);
      printf("\tMessage Type = %X\n", pkt->msgtype);
      printf("\tRaw Content = %s\n", rawpkt->content);
      fflush(stdout);
    }

    switch(pkt->service) {
      case YAHOO_SERVICE_USERSTAT:
      case YAHOO_SERVICE_CHATLOGON:
      case YAHOO_SERVICE_CHATLOGOFF:
      case YAHOO_SERVICE_LOGON:
      case YAHOO_SERVICE_LOGOFF:
      case YAHOO_SERVICE_ISAWAY:
      case YAHOO_SERVICE_ISBACK:
        process_packet_status(pkt);
        break;
      case YAHOO_SERVICE_MESSAGE:
      case YAHOO_SERVICE_CHATMSG:
      case YAHOO_SERVICE_SYSMESSAGE:
        process_packet_message(pkt);
        break;
      case YAHOO_SERVICE_NEWCONTACT:
        if(pkt->msg)
          process_packet_message(pkt);
        else
          process_packet_status(pkt);
        yahoo_get_config(context);
        break;
      case YAHOO_SERVICE_PING:
        process_packet_ping(pkt);
        break;
    }

    yahoo_free_packet(pkt);
    yahoo_free_rawpacket(rawpkt);
  }
}

